<?php

namespace App\Http\Controllers;

use App\Helpers;
use App\Http\Requests\InventoryStartDateRequest;
use App\Http\Requests\StoreWarehousePrioritySetting;
use App\Jobs\GenerateCacheDailyAverageConsumptionForProductsJob;
use App\Models\Product;
use App\Models\Setting;
use App\Repositories\InventoryMovementRepository;
use App\Repositories\ProductRepository;
use App\Repositories\SettingRepository;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SettingController extends Controller
{
    public function __construct(
        private readonly InventoryMovementRepository $inventoryMovementRepository,
        private readonly ProductRepository $productRepository,
        private readonly SettingRepository $settings,
    ) {
        parent::__construct();
    }

    /**
     * View All settings.
     */
    public function index(): JsonResponse
    {
        return $this->response->addData(Setting::all());
    }

    /**
     * Show an one setting.
     */
    public function show($id): JsonResponse
    {
        return $this->response->addData(Setting::with([])->findOrFail($id));
    }

    /**
     * Update setting.
     */
    public function update(Request $request, $id): JsonResponse
    {
        // update by "id" or "key"
        if (intval($id)) {
            $setting = Setting::with([])->findOrFail($id);
        } else {
            $setting = Setting::with([])->where('key', $id)->firstOrFail();
        }

        // validate type value
        #$request->validate(['value' => ($setting->isNullable() ? 'nullable' : 'required'). '|'.$setting->getCasts()['value']]);

        $setting->value = $request->input('value');
        $setting->save();

        return $this->response->addData($setting);
    }

    public function setWarehousePriority(StoreWarehousePrioritySetting $request): Response
    {
        $priority = $request->validated()['priority'];

        // Set the warehouse priority.
        $setting = Setting::with([])->updateOrCreate(
            ['key' => Setting::KEY_WAREHOUSE_PRIORITY],
            [
                'description' => 'The priority of warehouses.',
                'type' => Setting::TYPE_JSON,
                'value' => json_encode($priority),
                'default_value' => null,
            ]
        );

        return $this->response->addData($setting)
            ->setMessage(__('messages.settings.set_warehouse_priority'));
    }

    public function setInventoryStartDate(InventoryStartDateRequest $request): Response
    {
        $startDate = $request->validated()['date'];
        $setting = Setting::with([])->updateOrCreate(
            ['key' => Setting::KEY_INVENTORY_START_DATE],
            [
                'description' => 'The start date of inventory count.',
                'type' => Setting::TYPE_TIME,
                'value' => Helpers::dateLocalToUtc($startDate)->format('Y-m-d H:i:s'),
                'default_value' => null,
            ]
        );

        return $this->response->addData($setting)
            ->setMessage(__('messages.settings.set_inventory_start_date'));
    }

    public function updateDateSettingToUtc(Request $request): Response
    {
        $request->validate([
            'key' => 'required|string',
            'date' => 'nullable|date',
        ]);

        $date = $request->input('date');
        $setting = Setting::with([])->where('key', $request->input('key'))->firstOrFail();
        $setting->value = $date ? Helpers::dateLocalToUtc($request->input('date'))->format('Y-m-d H:i:s') : null;
        $setting->save();

        return $this->response->addData($setting)
            ->setMessage(__('messages.settings.update_date_setting_to_utc'));
    }

    /**
     * View setting for mapping nominal codes.
     */
    public function nominalCodesMapping(Request $request): Response
    {
        $input = $request->input('fetchByPaymentTypes');

        $settings = Setting::with([])->whereIn('key',
            $input == 'true' ? Setting::KEYS_PAYMENT_TYPES_MAPPING : Setting::KEYS_NC_MAPPING)->get();

        return $this->response->addData($settings);
    }

    /**
     * Update settings for mapping nominal codes.
     */
    public function updateNominalCodesMapping(Request $request): Response
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.id' => 'required_without:settings.*.key|exists:settings,id',
            'settings.*.key' => 'required_without:settings.*.id|exists:settings,key|in:'.implode(',',
                array_merge(Setting::KEYS_NC_MAPPING, Setting::KEYS_PAYMENT_TYPES_MAPPING)),
            'settings.*.value' => 'nullable|exists:nominal_codes,id',
        ]);

        $this->bulkUpdateNominalCodes($request->input('settings'));

        return $this->response->setMessage(__('messages.settings.update_nominal_code_mapping'));
    }

    /**
     * View settings for sales orders.
     */
    public function salesOrders(): Response
    {
        $settings = Setting::with([])->whereIn('key', Setting::KEYS_SO)->get();

        return $this->response->addData($settings);
    }

    /**
     * Update settings for sales orders.
     */
    public function updateSalesOrders(Request $request): Response
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.id' => 'required_without:settings.*.key|exists:settings,id',
            'settings.*.key' => 'required_without:settings.*.id|exists:settings,key|in:'.implode(',', Setting::KEYS_SO),
            'settings.*.value' => 'required',
        ]);

        $this->bulkUpdate($request->input('settings'));

        return $this->response->setMessage(__('messages.settings.update_sales_orders'));
    }

    public function updateAutoFulfillmentSettings(Request $request): Response
    {
        $request->validate([
            'fulfill_on_mapped_shipping_method' => 'required|boolean',
            'auto_fulfillment_failure_notifications' => 'required|boolean',
            'auto_fulfillment_email' => 'required_if:auto_fulfillment_failure_notifications,true|email',
        ]);

        $this->settings->updateAll([
            [
                'type' => Setting::TYPE_CHECKBOX,
                'key' => Setting::KEY_FULFILL_ON_MAPPED_SHIPPING_METHOD,
                'value' => $request->input('fulfill_on_mapped_shipping_method', false),
            ],
            [
                'type' => Setting::TYPE_CHECKBOX,
                'key' => Setting::KEY_AUTO_FULFILLMENT_FAILURE_NOTIFICATIONS,
                'value' => $request->input('auto_fulfillment_failure_notifications', false),
            ],
            [
                'type' => Setting::TYPE_EMAIL,
                'key' => Setting::KEY_AUTO_FULFILLMENT_EMAIL,
                'value' => $request->input('auto_fulfillment_email'),
            ],
        ]);

        return $this->response->setMessage(__('messages.settings.update_auto_fulfillment'));
    }

    /**
     * View settings for purchase orders.
     */
    public function purchaseOrders(): Response
    {
        $settings = Setting::with([])->whereIn('key', Setting::KEYS_PO)->get();

        return $this->response->addData($settings);
    }

    /**
     * Update settings for purchase orders.
     */
    public function updatePurchaseOrders(Request $request): Response
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.id' => 'required_without:settings.*.key|exists:settings,id',
            'settings.*.key' => 'required_without:settings.*.id|exists:settings,key|in:'.implode(',', Setting::KEYS_PO),
            'settings.*.value' => 'nullable',
        ]);

        $this->bulkUpdate($request->input('settings'));

        return $this->response->setMessage(__('messages.settings.update_purchase_orders'));
    }

    /**
     * View settings for inventory forecasting.
     */
    public function inventoryForecasting(): Response
    {
        $settings = Setting::with([])->whereIn('key', Setting::KEYS_INVENTORY_FORECASTING)->get();

        return $this->response->addData($settings);
    }

    /**
     * Update settings for inventory forecasting.
     */
    public function updateInventoryForecasting(Request $request): ?Response
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.id' => 'required_without:settings.*.key|exists:settings,id',
            'settings.*.key' => 'required_without:settings.*.id|exists:settings,key|in:'.implode(',',
                Setting::KEYS_INVENTORY_FORECASTING),
            'settings.*.value' => 'required',
        ]);

        $oldForecastingSettings = Setting::query()->whereIn('key', Setting::KEYS_INVENTORY_FORECASTING)->get();

        $this->bulkUpdate($request->input('settings'));

        $forecastingSettings = Setting::query()->whereIn('key', Setting::KEYS_INVENTORY_FORECASTING)->get();

        if (
            (
                $forecastingSettings->firstWhere('key', Setting::KEY_ADJUST_FORECAST_OUT_OF_STOCK)->value !=
                $oldForecastingSettings->firstWhere('key', Setting::KEY_ADJUST_FORECAST_OUT_OF_STOCK)->value
            ) ||
            (
                $forecastingSettings->firstWhere('key', Setting::KEY_DAYS_SALES_HISTORY)->value !=
                $oldForecastingSettings->firstWhere('key', Setting::KEY_DAYS_SALES_HISTORY)->value
            )
        ) {
            /*
             * Invalidate every product with movements in the past X days, with X being the days of sales history selected
             */
            $this->productRepository->invalidateDailyAverageConsumptionCache(
                $this->inventoryMovementRepository->getProductsWithMovementsInPastDays(
                    $forecastingSettings->firstWhere('key', Setting::KEY_DAYS_SALES_HISTORY)->value
                )->toArray()
            );
            dispatch(new GenerateCacheDailyAverageConsumptionForProductsJob())->onQueue('products');
        }

        return $this->response->setMessage(__('messages.settings.update_inventory_forecasting'));
    }

    /**
     * Bulk update settings.
     */
    private function bulkUpdate(array $settings)
    {
        foreach ($settings as $index => $settingInput) {
            if (! empty($settingInput['id'])) {
                $setting = Setting::with([])->findOrFail($settingInput['id']);
            } else {
                $setting = Setting::with([])->where('key', $settingInput['key'])->firstOrFail();
            }

            // validate type value
            $required = 'required';
            if (! is_null(@$settingInput['key']) && in_array($settingInput['key'], [
                'purchase_order_prefix',
                'purchase_order_cc_outgoing',
            ])) {
                $required = 'nullable';
            }

            $validator = Validator::make(['value' => $settingInput['value']],
                ['value' => $required.'|'.$setting->getCasts()['value']]);
            if ($validator->fails()) {
                $this->response->addWarningsFromValidator($validator, "settings.{$index}");

                continue;
            }

            $setting->value = $settingInput['value'];
            $setting->save();
        }
    }

    /**
     * Bulk update settings.
     */
    private function bulkUpdateNominalCodes(array $settings)
    {
        foreach ($settings as $index => $settingInput) {
            if (! empty($settingInput['id'])) {
                $setting = Setting::with([])->findOrFail($settingInput['id']);
            } else {
                $setting = Setting::with([])->where('key', $settingInput['key'])->firstOrFail();
            }

            $setting->value = $settingInput['value'] ?? null;
            $setting->save();
        }
    }

    public function getWarehouseLocationSettings(Request $request)
    {
        $settings = Setting::with([])->whereIn('key', Setting::KEYS_LOCATIONS)->get();

        return $this->response->addData($settings);
    }

    public function updateWarehouseLocationSettings(Request $request)
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.id' => 'required_without:settings.*.key|exists:settings,id',
            'settings.*.key' => 'required_without:settings.*.id|exists:settings,key|in:'.implode(',',
                Setting::KEYS_LOCATIONS),
            'settings.*.value' => 'nullable',
        ]);

        $this->bulkUpdate($request->input('settings'));

        return $this->response->setMessage(__('messages.settings.update_purchase_orders'));
    }

    /**
     * @throws Exception
     */
    public function updateBlemishedSkuPattern(Request $request): Response
    {
        $request->validate([
            'blemished_sku_pattern' => 'required|string',
            'use_blemished_sku_pattern' => 'required|boolean',
        ]);

        $this->settings->updateAll(
            [
                [
                    'key' => Setting::KEY_BLEMISHED_SKU_PATTERN,
                    'value' => $request->input('blemished_sku_pattern'),
                ],
                [
                    'key' => Setting::KEY_USE_BLEMISHED_SKU_PATTERN,
                    'value' => $request->input('use_blemished_sku_pattern'),
                ],
            ]
        );
        $generateExampleSkuPattern = (new Product())->generateSkuPattern();

        return $this->response->addData(['pattern' => $generateExampleSkuPattern]);
    }

    public function getBlemishedSkuPattern(): Response
    {
        $keys = [
            Setting::KEY_BLEMISHED_SKU_PATTERN,
            Setting::KEY_USE_BLEMISHED_SKU_PATTERN,
        ];
        $settings = Setting::with([])->whereIn('key', $keys)->get();

        return $this->response->addData($settings);
    }
}
