<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Resources\SalesOrderLineFinancialResource;
use App\Jobs\GenerateRecalculateSalesOrderLineFinancialsJob;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Response;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Illuminate\Database\QueryException;
use Illuminate\Http\Request;

class SalesOrderLineFinancialController extends Controller
{
    use DataTable;

    protected function getModel()
    {
        return SalesOrderLineFinancial::class;
    }

    protected function getResource()
    {
        return SalesOrderLineFinancialResource::class;
    }

    public function recalculateLines(Request $request): Response
    {
        set_time_limit(0);
        $this->validate($request, [
            'ids' => 'sometimes|array|min:1',
            'ids.*' => 'integer|exists:sales_order_lines,id',
            'filters' => 'sometimes|string',
        ]);

        if ($request->has('filters')) {
            try {
                $request->merge([
                        'ids' => $this->getIdsFromFilters(
                            modelClass: SalesOrderLineFinancial::class,
                            request: $request,
                            idField: 'sales_order_line_id'
                        )
                    ]
                );
            } catch (QueryException) {
                return $this->response->error(400)->addError('The filters are too complex to support select all results currently.  Please select one page at a time.', Response::HTTP_BAD_REQUEST, 'too_complex');
            }

            dispatch(new GenerateRecalculateSalesOrderLineFinancialsJob(SalesOrderLine::whereIn('id', $request->input('ids'))->get()));
            return $this->response->setMessage('Recalculations processing in the background');
        }

        (new SalesOrderLineFinancialManager())->recalculateLines(SalesOrderLine::whereIn('id', $request->input('ids'))->get());

        return $this->response->setMessage('Recalculation successful');
    }
}
