<?php

namespace App\Http\Controllers;

use App\Data\BulkActionData;
use App\Data\UpdateSalesOrderData;
use App\Data\UpdateSalesOrderPayloadData;
use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\CantUpdateNonExistingSalesOrderLineException;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\IntegrationInstance\UncancellableShippingProviderOrderException;
use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Exceptions\SalesOrder\LineFulfilledAtWarehouseException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Exceptions\SalesOrderFulfillmentReservationException;
use App\Exceptions\SalesOrderLineNotFulfillableException;
use App\Ghostscript\Ghostscript;
use App\Http\Controllers\Traits\BulkDeleteProcessor;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Controllers\Traits\SalesOrderSplitActions;
use App\Http\Requests\DropshipSalesOrderRequest;
use App\Http\Requests\FulfillFBASalesOrderRequest;
use App\Http\Requests\FulfillSalesOrderRequest;
use App\Http\Requests\ImportCSVFileRequest;
use App\Http\Requests\ResendSalesOrderRequest;
use App\Http\Requests\SalesOrderRequest;
use App\Http\Resources\PurchaseOrderResource;
use App\Http\Resources\SalesOrderFinancialResource;
use App\Http\Resources\SalesOrderResource;
use App\Jobs\ExportSalesOrderPackingSlipJob;
use App\Jobs\GenerateSalesOrderPackingSlipsJob;
use App\Models\OrderLink;
use App\Models\PackingSlipQueue;
use App\Models\PaymentType;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\ShippingProvider;
use App\Models\ShipStation\ShipstationOrder;
use App\Observers\AddPackingSlipQueueObserver;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Response;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use App\Services\SalesOrder\ApproveSalesOrderService;
use App\Services\SalesOrder\BulkApproveSalesOrderService;
use App\Services\SalesOrder\Fulfillments\BulkFulfillmentManager;
use App\Services\SalesOrder\FulfillSalesOrderService;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\StockTake\OpenStockTakeException;
use App\Validators;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Modules\Amazon\Exceptions\AmazonFulfillmentOrderException;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Response as ResponseAlias;
use Throwable;

class SalesOrderController extends Controller
{
    use BulkDeleteProcessor, BulkOperation, DataTable, ImportsData, SalesOrderSplitActions;

    protected string $model_path = SalesOrder::class;

    public function __construct(
        protected readonly SalesOrderManager $manager,
        protected readonly BulkFulfillmentManager $fulfillmentManager,
        protected readonly SalesOrderRepository $orders,
    ) {
        parent::__construct();
    }

    /**
     * Get a Sales Order by ID.
     */
    public function show($id): Response
    {
        /**
         * Get Sales Order by id with relations Or fail 404 not found.
         */
        $salesOrder = SalesOrder::with([
            'payments',
            'notes',
            'purchaseOrders',
            'salesOrderLines.product.productInventory',
            'salesOrderLines.backorderQueue',
            'adjustments.product',
            'salesOrderLines.inventoryMovements',
            'salesOrderFulfillments.salesOrderFulfillmentLines',
            'financialLines.financialLineType',
            'accountingTransaction.accountingIntegration',
            'salesOrderLines.adjustments.product',
            'customFieldValues.customField',
            'shippingMethod',
            'backorderedLines.backorderQueue.backorderQueueCoverages.purchaseOrderLine.purchaseOrder'
        ])->addRelations()->findOrFail($id);

        return $this->response->addData(SalesOrderResource::make($salesOrder));
    }

    /**
     * Get a Sales Order Document by ID.
     */
    public function showDocument(SalesOrder $salesOrder): Response
    {
        $document = $salesOrder->order_document;

        return $this->response->addData($document);
    }

    /**
     * Creates a sales order through the manager.
     *
     * @throws Throwable
     */
    public function store(SalesOrderRequest $request): Response
    {
        try {
            $salesOrder = $this->manager
                ->createOrder($request->validated())
                ->load($this->getRequiredRelations());

            return $this->response->success(ResponseAlias::HTTP_CREATED)
                ->setMessage(__('messages.success.create', ['resource' => 'sales order']))
                ->addData(SalesOrderResource::make($salesOrder));
        } catch (InvalidProductWarehouseRouting $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage())
                ->setErrors(['resource' => 'product', 'sku' => $e->product->sku]);
        } catch (OpenStockTakeException $openStockTakeException) {
            return $this->response->error($openStockTakeException->status)
                ->setMessage($openStockTakeException->getMessage())
                ->setErrors(['resource' => 'stock take', 'id' => $openStockTakeException->stockTake->id]);
        }
    }

    /**
     * @throws Throwable
     */
    public function update(UpdateSalesOrderPayloadData $payload, SalesOrder $salesOrder): Response
    {
        $data = UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder->load($this->getRequiredRelations()),
            'payload' => $payload,

        ]);
        try {
            $salesOrder = $this->manager
                ->updateOrder($data)
                ->load($this->getRequiredRelations());

            return $this->response->success(ResponseAlias::HTTP_OK)
                ->setMessage(__('messages.success.update', ['resource' => 'sales order']))
                ->addData(SalesOrderResource::make($salesOrder));
        } catch (InvalidProductWarehouseRouting $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage())
                ->setErrors(['resource' => 'product', 'sku' => $e->product->sku]);
        } catch (OpenStockTakeException $openStockTakeException) {
            return $this->response->error($openStockTakeException->status)
                ->setMessage($openStockTakeException->getMessage())
                ->setErrors(['resource' => 'stock take', 'id' => $openStockTakeException->stockTake->id]);
        } catch (LineFulfilledAtWarehouseException $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage())
                ->setErrors(['resource' => 'product', 'sku' => $e->product->sku]);
        } catch (CantUpdateNonExistingSalesOrderLineException $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage())
                ->setErrors(['resource' => 'sales order line']);
        }
    }

    /**
     * @throws Throwable
     */
    public function archive(FulfillSalesOrderRequest $request, $salesOrderId): Response
    {
        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::with([])->findOrFail($salesOrderId);

        if ($salesOrder->archive()) {
            set_time_limit(0);
            if (is_array($fulfillment = FulfillSalesOrderService::make($salesOrder)->fulfill($request, true))) {
                return $this->response->error(Response::HTTP_BAD_REQUEST)->addError(...$fulfillment);
            }

            $this->response->addData($fulfillment->only(['id', 'fulfillment_sequence', 'ssi_imported']));

            return $this->response->setMessage(__('messages.success.create', ['resource' => 'sales order fulfillment']));
        }

        return $this->response->addWarning(__('messages.failed.already_archive'), 'SalesOrder'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $salesOrderId]);
    }

    public function unarchived($salesOrderId)
    {
        $salesOrder = SalesOrder::with([])->findOrFail($salesOrderId);

        if ($salesOrder->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'sales order',
                    'id' => $salesOrder->id,
                ]))
                ->addData(SalesOrderResource::make($salesOrder));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'sales order',
                'id' => $salesOrder->id,
            ]), 'SalesOrder'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $salesOrder->id])
            ->addData(SalesOrderResource::make($salesOrder));
    }

    /**
     * Create dropship request(create a new purchase order).
     */
    public function dropshipRequest(DropshipSalesOrderRequest $request, SalesOrder $salesOrder): Response
    {
        $purchaseOrder = ApproveSalesOrderService::make($salesOrder)->approveDropshipRequest($request);

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'dropship request']))->addData(PurchaseOrderResource::make($purchaseOrder));
    }

    /**
     * Fulfill sales order.
     *
     *
     * @throws Throwable
     */
    public function fulfill(FulfillSalesOrderRequest $request, SalesOrder $salesOrder): Response
    {
        try {
            $inputs = $request->validated();
            /*
             * By default fulfillments get submitted to the shipping provider and sales channel.  The following
             * allows the frontend control over whether to submit the fulfillment to the shipping provider
             * and sales channel.
             */
            $submitToShippingProvider = (! isset($inputs['submit_to_shipping_provider']) || $inputs['submit_to_shipping_provider'] === true);
            unset($inputs['submit_to_shipping_provider']);
            $submitToSalesChannel = (! isset($inputs['submit_to_sales_channel']) || $inputs['submit_to_sales_channel'] === true);
            unset($inputs['submit_to_sales_channel']);

            $fulfillment = $this->fulfillmentManager->fulfill($salesOrder, $request->validated(), $submitToShippingProvider, $submitToSalesChannel);

            return $this->response->success(ResponseAlias::HTTP_CREATED)
                ->setMessage(__('messages.success.create', ['resource' => 'sales order']))
                ->addData($fulfillment?->only(['id']));
        } catch (SalesOrderFulfillmentException|UncancellableShippingProviderOrderException $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage())
                ->setErrors($e->errors);
        } catch (SalesOrderFulfillmentReservationException $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage().$e->salesOrderLine?->product ? ' SKU: '.$e->salesOrderLine?->product->sku.' is out of stock or has an inventory integrity issue' : '')
                ->setErrors([
                    'resource' => 'fulfillment',
                    'id' => $e->fulfillment?->id,
                    'product' => $e->salesOrderLine?->product?->only(['id', 'sku']),
                ]);
        } catch (SalesOrderLineNotFulfillableException|SalesOrderFulfillmentDispatchException $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage())
                ->setErrors([
                    'resource' => 'fulfillment'
                ]);
        } catch (AmazonFulfillmentOrderException $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage());
        } catch (OpenStockTakeException $e) {
            return $this->response->error($e->status)
                ->setMessage($e->getMessage())
                ->setErrors(['resource' => 'stock take', 'id' => $e->stockTake->id]);
        } catch (InsufficientStockException $e) {
            return $this->response->error(ResponseAlias::HTTP_BAD_REQUEST)
                ->setMessage($e->getMessage())
                ->setErrors([
                    'resource' => 'product',
                    'sku' => $e->productSku,
                    'product_id' => $e->productId
                ]);
        }
    }

    /**
     * @throws BindingResolutionException
     */
    public function compareFulfillmentsWithSalesChannel(Request $request, SalesOrder $salesOrder)
    {
        /*
         * TODO: right now since SalesOrder model is missing a relationship with a salesChannelOrder, we will just use
         *  a conditional for Shopify
         */
        if ($salesOrder->shopifyOrder) {
            return $this->response->success()->addData(
                $salesOrder->shopifyOrder->compareFulfillments()
            );
        }

        return $this->response->success();
    }

    /**
     * Bulk approve sales orders.
     */
    public function bulkApprove(Request $request): Response
    {
        $request->validate(['ids' => 'required_without:filter|array', 'filters' => 'required_without:ids']);

        $bulkApproveService = new BulkApproveSalesOrderService($request);

        $response = $bulkApproveService->approve();
        $approveCount = count(Arr::flatten($response, 1));
        if ($approveCount == 0) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.bulk_so_approve_empty'));
        }
        // all sales orders failed to approve
        if ($approveCount == $bulkApproveService->getErrorsCount()) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.bulk_so_approve'))->setErrors($response);
        }
        // some sales orders failed to approve
        if ($bulkApproveService->getErrorsCount()) {
            return $this->response->warning()
                ->setMessage(__('messages.sales_order.not_all_so_approved', [
                    'success' => ($approveCount - $bulkApproveService->getErrorsCount()),
                    'total' => $approveCount,
                ]))->setWarnings($response);
        }

        // all sales orders approved successfully
        return $this->response->setMessage(__('messages.success.bulk_so_approve'))->addData($response);
    }

    /**
     * Bulk fulfill sales orders.
     */
    public function bulkFulfill(BulkActionData $data): Response
    {
        try{
            $results = $this->fulfillmentManager->fulfillBulkOrders($data);
            if($results->successful()) {
                return $this->response->setMessage(__('messages.success.bulk_so_fulfill'))->addData($results);
            } else {
                return $this->response->warning()
                    ->setMessage(__('messages.sales_order.not_all_so_fulfilled', [
                        'success' => $results->success,
                        'total' => $results->total,
                    ]))->setWarnings($results->errors);
            }
        }catch (Throwable $e){
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage($e->getMessage());
        }
    }

    /**
     * Fulfill FBA sales order.
     *
     *
     * @throws Throwable
     */
    public function fulfillFBA(FulfillFBASalesOrderRequest $request, SalesOrder $salesOrder): Response
    {
        if (config('app.env') != 'production') {
            return $this->response->addError('Only available on production.', Response::CODE_UNACCEPTABLE, 'FulfillFBA');
        }

        FulfillSalesOrderService::make($salesOrder)->fulfillFBA($request);

        return $this->response;
    }

    public function shipstationOrder(SalesOrder $salesOrder)
    {
        $fulfillmentIds = $salesOrder->salesOrderFulfillments()->pluck('id');
        $order = ShipstationOrder::with([])->whereIn('sku_fulfillment_id', $fulfillmentIds)->get();

        return $this->response->addData($order);
    }

    public function shipstationFulfillment($salesOrderFulfillmentId)
    {
        $order = ShipstationOrder::with([])->where('sku_fulfillment_id', (int) $salesOrderFulfillmentId)->firstOrFail();

        return $this->response->addData($order);
    }

    /**
     * Constants we need it in forms.
     */
    public function constants(): JsonResponse
    {
        return $this->response->addData([
            'sales_order_status' => SalesOrder::STATUSES,
            'shipping_provider_types' => ShippingProvider::TYPES,
        ]);
    }

    /**
     * Remove sales order.
     *
     *
     * @throws Exception
     * @throws Throwable
     */
    public function destroy(SalesOrder $salesOrder): Response
    {
        $this->orders->bulkDelete(Arr::wrap($salesOrder->id));

        return $this->response->setMessage(__(
            'messages.success.delete',
            [
                'resource' => 'sales order',
                'id' => $salesOrder->sales_order_number,
            ]
        ));
    }

    /**
     * bulk delete using request filters or body ids array.
     *
     *
     * @throws Exception|Throwable
     */
    public function bulkDestroy(Request $request): Response
    {
        $inputs = $request->validate([
            'filters' => 'required_without:ids',
            'ids' => 'required_without:filters|array|min:1',
            'ids.*' => 'integer|exists:sales_orders,id',
        ]);

        $ids = $inputs['ids'] ?? $this->getIdsFromFilters(SalesOrder::class, $request);

        $this->orders->bulkDelete($ids);

        return $this->response->setMessage(__('messages.success.bulk_delete', ['resource' => 'sales orders']));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:sales_orders,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $salesOrders = SalesOrder::with([])->whereIn('id', $ids)->select('id', 'sales_order_number')->get();
        foreach ($salesOrders as $index => $salesOrder) {
            $isUsed = $salesOrder->isUsed();
            /*
             * Going to mark isUsed false since we take care of the logic to delete relate entities
             * anyway
             */
            $isUsed = false;

            $result[$index] = $salesOrder->only('id', 'sales_order_number');
            $result[$index]['deletable'] = $isUsed ? false : true;
            $result[$index]['reason'] = $isUsed ?: null;

            // Add warning for fulfillments if available. SKU-2487
            $fulfillmentCount = $salesOrder->salesOrderFulfillments()->count();
            if ($fulfillmentCount > 0) {
                $result[$index]['warnings']['fulfillments'] = trans_choice('messages.currently_used', $fulfillmentCount, [
                    'resource' => 'sales order fulfillment',
                    'model' => 'sales order ('.$salesOrder->sales_order_number.')',
                ]);
            }
        }

        return $this->response->addData($result);
    }

    /**
     * Get next customer reference for local channel (sku.io).
     *
     * @return string
     */
    public function getNextLocalNumber()
    {
        return $this->response->addData(SalesOrder::getNextLocalNumber(), 'next_customer_reference');
    }

    public function duplicate(Request $request, SalesOrder $salesOrder): Response
    {
        $newSalesOrder = app(SalesOrderManager::class)->duplicate($salesOrder);

        $this->response->setMessage(__('messages.sales_order.duplicate_success'));

        return $this->show($newSalesOrder->id);
    }

    /**
     * Resend sales order.
     */
    public function resend(ResendSalesOrderRequest $request, SalesOrder $salesOrder): Response
    {
        $newSalesOrder = $this->resendExchange($salesOrder, $request->input('sales_order_lines', []), true);

        // link resend sales order with original sales order
        $orderLink = new OrderLink();
        $orderLink->child = $newSalesOrder;
        $orderLink->link_type = OrderLink::LINK_TYPE_RESEND;
        $salesOrder->childLinks()->save($orderLink);

        $this->response->setMessage(__('messages.sales_order.resend_success'));

        return $this->show($newSalesOrder->id);
    }

    /**
     * Exchange sales order.
     */
    public function exchange(ResendSalesOrderRequest $request, SalesOrder $salesOrder): Response
    {
        $newSalesOrder = $this->resendExchange($salesOrder, $request->input('sales_order_lines', []));
        $newSalesOrder->payments()->create([
            'payment_date' => now(),
            'payment_type_id' => PaymentType::with([])->where('name', PaymentType::SALES_CREDIT_PAYMENT_TYPE_NAME)->value('id'),
            'amount' => $newSalesOrder->total,
            'currency_id' => $newSalesOrder->currency_id,
        ]);
        $newSalesOrder->setPaymentStatus();
        $newSalesOrder->approve();
        if ($request->has('sales_credit_id')) {
            $salesCredit = SalesCredit::with([])->find($request->input('sales_credit_id'));

            $orderLink = new OrderLink();
            $orderLink->child = $newSalesOrder;
            $orderLink->link_type = OrderLink::LINK_TYPE_EXCHANGE;
            $salesCredit->childLinks()->save($orderLink);
        } else {
            // link exchanged sales order with original sales order
            $orderLink = new OrderLink();
            $orderLink->child = $newSalesOrder;
            $orderLink->link_type = OrderLink::LINK_TYPE_EXCHANGE;
            $salesOrder->childLinks()->save($orderLink);
        }

        $this->response->setMessage(__('messages.sales_order.exchange_success'));

        return $this->show($newSalesOrder->id);
    }

    /**
     * Duplicate sales order to resend or exchange.
     * @throws Throwable
     */
    private function resendExchange(SalesOrder $salesOrder, array $orderLines, bool $resend = false): SalesOrder
    {
        return DB::transaction(function () use ($salesOrder, $orderLines, $resend) {

            // duplicate sales order
            $clonedSalesOrder = $salesOrder->replicate([
                'total',
                'archived_at',
                'order_status',
                'fulfillment_status',
                'payment_status',
                'fulfilled_at',
                'total_revenue',
                'fully_paid_at',
            ]);

            $clonedSalesOrder->sales_order_number = $salesOrder->sales_order_number . '-' . ($resend ? 'R' : 'E') . '-' . now()->format('Ymd');
            $clonedSalesOrder->order_status = SalesOrder::STATUS_DRAFT;
            $clonedSalesOrder->order_date = now();

            if ($resend) {
                $shipBy = now()->addDays(Setting::getValueByKey(Setting::KEY_SO_DEFAULT_HANDLING_DAYS));

                $cutOff = Setting::getValueByKey(Setting::KEY_SO_SAME_DAY_SHIPPING_CUTOFF);
                $hours = explode(':', $cutOff)[0];
                $minutes = explode(':', $cutOff)[1];

                if ($shipBy->isAfter(now()->addDays(Setting::getValueByKey(Setting::KEY_SO_DEFAULT_HANDLING_DAYS))->setTime($hours, $minutes))) {
                    $shipBy = $shipBy->addDay()->setTime(0, 0);
                }
                if ($shipBy->isSaturday()) {
                    $shipBy = $shipBy->addDays(2);
                }
                if ($shipBy->isSunday()) {
                    $shipBy = $shipBy->addDay();
                }
                $clonedSalesOrder->ship_by_date = $shipBy;
                if (isset($clonedSalesOrder->shippingMethod->delivery_max)) {
                    $clonedSalesOrder->deliver_by_date = $shipBy->addDays($clonedSalesOrder->shippingMethod->delivery_max);
                }
            }

            $clonedSalesOrder->save();

            // duplicate sales order lines with a new quantity
            foreach ($orderLines as $line) {
                /** @var SalesOrderLine $newSalesOrderLine */
                $newSalesOrderLine = $salesOrder->salesOrderLines->firstWhere('id', $line['id'])
                    ->replicate([
                        'discount',
                        'tax',
                        'split_from_line_id',
                        'has_backorder',
                        'fulfilled_quantity'
                    ]);
                $newSalesOrderLine->fulfilled_quantity = 0;
                $newSalesOrderLine->quantity = $line['quantity'];
                $newSalesOrderLine->amount = $resend ? 0 : $newSalesOrderLine->amount;
                $clonedSalesOrder->salesOrderLines()->save($newSalesOrderLine);
            }

            $clonedSalesOrder->save(); // to recalculate total

            return $clonedSalesOrder;
        });
    }

    /**
     * Import sales orders from CSV file.
     *
     *
     * @throws Throwable
     */
    public function importCSV(ImportCSVFileRequest $request): JsonResponse
    {
        // sales orders need to adding/updating Lines and approve it if order status changed to Open
        $draftSalesOrders = [];
        $successCount = 0;

        foreach ($request->getRows() as $index => $row) {
            // apply validation rules on the row
            $validator = Validators::salesOrderLineFromCSV($row);
            if ($validator->fails()) {
                $this->response->addWarningsFromValidator($validator, "sales_orders.{$index}");

                continue;
            }

            // get or create a new sales order
            // we use this check to prevent make useless query
            $salesOrder = empty($row['sales_order_id']) ? new SalesOrder() : SalesOrder::with([])->findOrFail($row['sales_order_id']);
            $salesOrder->fill($row);

            // skip the existing sales order if override data is false
            if ($salesOrder->exists && ! $request->overrideData()) {
                continue;
            }

            // adding addresses/lines for a new sales order and for a draft sales order
            if ($salesOrder->order_status == SalesOrder::STATUS_DRAFT) {
                $salesOrder->setCustomer($request->getElementsByPrefix($row, 'customer_'));
                $salesOrder->setShippingAddress($request->getElementsByPrefix($row, 'shipping_'));
                $salesOrder->setBillingAddress($request->getElementsByPrefix($row, 'billing_'));

                $salesOrder->save();

                // add to draft sales order to adding/updating lines and approve it if order status changed to Open
                if (isset($draftSalesOrders[$salesOrder->id])) {
                    $draftSalesOrders[$salesOrder->id]['lines'][] = $request->getElementsByPrefix($row, 'order_line_');
                } else {
                    $draftSalesOrders[$salesOrder->id] = [
                        'index' => $index,
                        'id' => $salesOrder->id,
                        'order_status' => $row['order_status'],
                        'lines' => [$request->getElementsByPrefix($row, 'order_line_')],
                    ];
                }
            } else {
                // save basic sales order data(like shipping_method, fulfillment date...)
                $salesOrder->save();
            }

            $successCount++;
        }

        // add/update sales order lines and approve the draft sales order if order status changed to Open
        // we add sales order lines after fetching whole file to sync sales order lines (create/update/deletes)
        foreach (array_chunk($draftSalesOrders, 20) as $orders) {
            $orderIds = array_column($orders, 'id');
            /** @var SalesOrder $salesOrder */
            foreach (SalesOrder::with([])->findMany($orderIds) as $salesOrder) {
                $order = collect($orders)->firstWhere('id', $salesOrder->id);

                // syncing sales order lines
                if ($order['order_status'] != SalesOrder::STATUS_CLOSED) {
                    $salesOrder->setSalesOrderLines($order['lines']);
                }

                // approve sales order if order status changed to Open
                if ($order['order_status'] != SalesOrder::STATUS_DRAFT) {
                    $approve = $salesOrder->approve($order['order_status'] == SalesOrder::STATUS_CLOSED);
                    if (is_array($approve)) { // complete validation errors
                        $this->response->addWarningsFromValidator($approve, "sales_orders.{$order['index']}");
                    }
                }

                if ($order['order_status'] == SalesOrder::STATUS_CLOSED) {
                    $salesOrder->fullyFulfill($inputs['tracking_number'] ?? null);
                }
            }
        }

        // at least one sales order added successfully
        if ($successCount) {
            return $this->response->setMessage(__('messages.success.import', ['resource' => 'sales orders']));
        } else {
            return $this->response->setMessage(__('messages.failed.csv_all_lines_incorrect'));
        }
    }

    /**
     * Add note to sales order.
     */
    public function addNote(Request $request, SalesOrder $salesOrder): Response
    {
        $request->validate(['note' => 'required']);

        $note = $salesOrder->notes()->create($request->only('note'));
        $note->load('user');

        // reprint the packing slips
        AddPackingSlipQueueObserver::salesOrderUpdated($salesOrder);

        return $this->response->addData($note)
            ->setMessage(__('messages.success.create', ['resource' => 'note']));
    }

    /**
     * View sales order notes.
     */
    public function notes(SalesOrder $salesOrder): JsonResponse
    {
        return $this->response->setData($salesOrder->notes()->with('user')->orderByDesc('created_at')->paginate()->toArray());
    }

    public function financialsProforma($salesOrderId): Response
    {
        $manager = new SalesOrderLineFinancialManager(Arr::wrap($salesOrderId));
        $manager->calculate();

        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::with([
            'salesOrderLineFinancials',
            'salesOrderLineFinancials.salesOrderLine',
            'salesOrderLineFinancials.salesOrderLine.product',
        ])->findOrFail($salesOrderId);

        return $this->response->addData(SalesOrderFinancialResource::make($salesOrder));
    }

    /**
     * Deletes a sales order note.
     */
    public function deleteNote(SalesOrder $salesOrder, $noteId): Response
    {
        $note = $salesOrder->notes()->find(e($noteId));
        if ($note) {
            $note->delete();
        }

        // reprint the packing slips
        AddPackingSlipQueueObserver::salesOrderUpdated($salesOrder);

        return $this->response->setMessage(__('messages.success.delete', ['resource' => 'note', 'id' => $noteId]));
    }

    /**
     * Export packing slips of sales orders.
     *
     *
     * @throws Exception
     */
    public function exportPackingSlips(Request $request): Response
    {
        $request->validate([
            'ids' => 'required_without:filters',
            'filters' => 'required_without:ids',
        ]);

        if($request->has('filters')){
            dispatch(new ExportSalesOrderPackingSlipJob($request->user(), $request->get('filters')));
        }

        $ids = explode(',', $request->get('ids'));
        $printedOrderIds = $ids;
        $needsPrinting = [];
        foreach ($ids as $id) {
            // the file does not exist
            if (! Storage::disk('order_packing_slips')->exists("$id.pdf")) {
                $needsPrinting[] = $id;
                AddPackingSlipQueueObserver::addPackingSlipQueue((new SalesOrder())->forceFill(['id' => (int) $id]), false);
            }
            // generating the packing slip still in-progress (in the queue)
            elseif (PackingSlipQueue::query()->firstWhere(['link_id' => (int) $id, 'link_type' => SalesOrder::class])) {
                $needsPrinting[] = $id;
            }
        }

        // print unprinted orders in one file
        if (! empty($needsPrinting)) {
            $generatingPackingSlipResponse = (new GenerateSalesOrderPackingSlipsJob($needsPrinting, null, ['one_file' => true]))->handle();
            if (! empty($generatingPackingSlipResponse['errors'])) {
                $this->response->setWarnings($generatingPackingSlipResponse['errors']);
                $ids = array_diff($ids, array_keys($generatingPackingSlipResponse['errors']));
                $printedOrderIds = $ids;
            }
            if (! empty($generatingPackingSlipResponse['generatedFile'])) {
                $unprintedFile = str_replace([Storage::disk('order_packing_slips')->path(''), '.pdf'], '', $generatingPackingSlipResponse['generatedFile']);
                $ids = array_merge(array_diff($ids, $needsPrinting), [$unprintedFile]);
            }
            $ids = array_values($ids);
        }

        if (empty($ids)) {
            return $this->response->setMessage('There are no packing slips for the selected orders.');
        }

        // mark sales orders as printed
        SalesOrder::query()->whereIn('id', $printedOrderIds)->update(['packing_slip_printed_at' => now()]);

        if (count($ids) === 1) {
            return $this->response->setMessage('The packing slip file has been generated successfully')
                ->addData(['file' => Storage::disk('order_packing_slips')->url("{$ids[0]}.pdf")]);
        }

        $outputFile = time().rand(1000, 9999).'.pdf';
        $psFiles = array_map(fn ($id) => Storage::disk('order_packing_slips')->path("$id.pdf"), $ids);
        if (is_null($combineStatus = (new Ghostscript(Storage::disk('reports')->path($outputFile), $psFiles))->combine())) {
            return $this->response->setMessage('The packing slip file has been generated successfully')
                ->addData(['file' => Storage::disk('reports')->url($outputFile)]);
        }

        throw new Exception("Ghostscript error: {$combineStatus}");
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return SalesOrder::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return SalesOrderResource::class;
    }

    /**
     * Get Required Relations.
     */
    protected function getRequiredRelations(): array
    {
        $baseRelations = DataTableConfiguration::getRequiredRelations(SalesOrder::class);

        $relationsForView = [
            'shippingAddress',
            'billingAddress',
            'salesOrderLines.product.totalInventory',
            'salesOrderLines.product.productInventory',
            'salesOrderLines.product.suppliersInventory',
            'salesOrderLines.product.primaryImage',
            'salesOrderLines.salesCreditLines',
            'salesOrderLines.inventoryMovements',
            'salesOrderLines.salesOrderFulfillmentLines',
            'salesOrderLines.backorderQueue',
            'purchaseOrders',
            'salesCredits',
            'salesCreditsParentsLinks.parent',
            'salesCreditsParentsLinks.parent.parentLinks',
            'customFieldValues',
        ];

        return array_merge($baseRelations, $relationsForView);
    }

    public function downloadInvoicePDF(SalesOrder $salesOrder): BinaryFileResponse
    {
        return response()->download($salesOrder->exportToPDF());
    }

    /**
     * Mark the sales order as printed
     */
    public function markAsPrinted($salesOrderId): Response
    {
        SalesOrder::where('id', $salesOrderId)->update(['packing_slip_printed_at' => now()]);

        return $this->response->setMessage(__('messages.sales_order.marked_as_printed'));
    }

    public function getOrdersHaveUnallocatedItem($salesOrderId, $productId)
    {
        $salesOrders = SalesOrderLine::with(['product', 'backorderQueue', 'salesOrder'])
            ->whereNot('id', $salesOrderId)
            ->whereHas('product', function ($q) use ($productId) {
                $q->where('id', $productId);
            })
            ->whereHas('backorderQueue')
            ->get()->filter(function ($line) {
                return $line->backorderQueue->unallocated_backorder_quantity > 0;
            })
            ->pluck('salesOrder')
            ->unique()
            ->map(function ($salesOrder) {
                return [
                    'id' => $salesOrder->id,
                    'sales_order_number' => $salesOrder->sales_order_number,
                ];
            });

        return $this->response->addData($salesOrders);
    }
}
