<?php

namespace App\Http\Controllers;

use App\Http\Requests\PaidSalesCreditRequest;
use App\Http\Resources\SalesCreditResource;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\SalesCredit;
use App\Services\Payments\Payable;
use Illuminate\Support\Facades\DB;

/**
 * Class SalesCreditPaymentController.
 */
class SalesCreditPaymentController extends SalesCreditController
{
    /**
     * @return mixed
     */
    public function updatePayment(PaidSalesCreditRequest $request, $salesCreditId, $paymentId)
    {
        $payload = $request->validated();

        $salesCredit = SalesCredit::with(['payments', 'allocations'])->findOrFail(e($salesCreditId));
        $payment = $this->getSalesCreditPayment($salesCredit, $paymentId);

        return DB::transaction(function () use ($salesCredit, $payment, $payload) {
            // Credit payment types cannot be changed as they are system created
            if (isset($payload['amount'])) {
                $payload['amount'] = -abs($payload['amount']);
            }
            if ($payment->paymentType->name === PaymentType::SALES_CREDIT_PAYMENT_TYPE_NAME) {
                unset($payload['payment_type_id']);
            }
            $payment->fill($payload);
            $payment->save();

            // Update sales credit payment status
            if ($payment->link instanceof Payable) {
                $payment->link->setPaymentStatus(now());
            }

            $salesCredit->load(['payments', 'allocations']);
            $this->response->addData(SalesCreditResource::make($salesCredit->load($this->getRequiredRelations())));

            return $this->response->setMessage(__('messages.success.create', ['resource' => 'payment']));
        });
    }

    /**
     * @return mixed
     */
    public function deletePayment($salesCreditId, $paymentId)
    {
        $salesCredit = SalesCredit::with(['payments', 'allocations'])->findOrFail(e($salesCreditId));
        $payment = $this->getSalesCreditPayment($salesCredit, $paymentId);

        return DB::transaction(function () use ($payment, $salesCredit) {
            $payment->delete();

            // Update sales credit payment status
            if ($payment->link instanceof Payable) {
                $payment->link->setPaymentStatus(now());
            }

            // Delete any Sales Credit Allocation attached to the payment
            $salesCredit->load(['payments', 'allocations']);

            $this->response->addData(SalesCreditResource::make($salesCredit->load($this->getRequiredRelations())));

            return $this->response->setMessage(__('messages.success.delete', ['resource' => 'payment']));
        });
    }

    private function getSalesCreditPayment(SalesCredit $salesCredit, $paymentId): Payment
    {
        $payment = $salesCredit->payments()->find(e($paymentId));

        return $payment;
    }
}
