<?php

namespace App\Http\Controllers;

use App\Data\SalesCreditData;
use App\Data\UpdateSalesCreditData;
use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\PaidSalesCreditRequest;
use App\Http\Requests\ReceiveSalesCreditRequest;
use App\Http\Requests\StoreSalesCredit;
use App\Http\Resources\SalesCreditResource;
use App\Jobs\GenerateSalesCreditInvoice;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\Payment;
use App\Models\SalesCredit;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Notifications\SalesCreditIssuedNotification;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Throwable;

class SalesCreditController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = SalesCredit::class;

    /**
     * View sales credit.
     */
    public function show(SalesCredit $salesCredit): Response
    {
        $salesCredit->load($this->getRequiredRelations());

        return $this->response->addData(SalesCreditResource::make($salesCredit));
    }

    public function previewInvoice(SalesCredit $salesCredit): Response
    {
        $path = dispatch_sync(new GenerateSalesCreditInvoice($salesCredit));
        $path = Storage::disk('reports')->url($path);

        return $this->response->addData(compact('path'));
    }

    /**
     * Create a new Sales Credit.
     *
     * TODO: Change StoreSalesCredit to Data object
     * @throws Throwable
     */
    public function store(SalesCreditData $request): Response
    {
        $inputs = $request->toArray();

        DB::transaction(function () use ($inputs) {
            if (! isset($inputs['currency_id']) && ! isset($inputs['currency_code'])) {
                // Use default currency id
                $inputs['currency_id'] = Currency::default()->id;
            }

            // Create sales credit
            $salesCredit = new SalesCredit($inputs);
            $salesOrder = null;

            if (! empty($inputs['sales_order_id'])) {
                $salesOrder = SalesOrder::with([])->findOrFail($inputs['sales_order_id']);
                $salesOrder->addSalesCredit($salesCredit);
            } else {
                $customer = Customer::with([])->findOrFail(e($inputs['customer_id']));
                $customer->addSalesCredit($salesCredit);
            }

            // add sales credit lines to the sales credit
            $salesCredit->setSalesCreditLines($inputs['sales_credit_lines']);
            $salesCredit->setTotal();
            $salesCredit->setTaxAllocation();

            if ($salesOrder) {
                $salesOrder->setPaymentStatus();
            }

            // Notify the customer about the sales credit
            //      $salesCredit->customer->notify(new SalesCreditIssuedNotification($salesCredit));

            $this->response->addData(SalesCreditResource::make($salesCredit));
        });

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'sales credit']));
    }

    /**
     * Update Sales Credit.
     *
     *
     * @throws Throwable
     */
    public function update(UpdateSalesCreditData $request, SalesCredit $salesCredit): Response
    {

        $inputs = $request->toArray();
        DB::transaction(function () use ($salesCredit, $inputs) {
            // update sales credit
            $salesCredit->fill($inputs);
            $salesCredit->save();

            // sync sales credit lines
            $salesCredit->setSalesCreditLines(array_key_exists('sales_credit_lines', $inputs) ? $inputs['sales_credit_lines'] : false);
            $salesCredit->setTaxAllocation();

            $this->response
                ->setMessage(__('messages.success.update', [
                    'resource' => 'sales credit',
                    'id' => $salesCredit->sales_credit_number,
                ]))
                ->addData(SalesCreditResource::make($salesCredit->load($this->getRequiredRelations())));
        });

        return $this->response;
    }

    /**
     * Receive sales credit (return).
     *
     *
     * @throws Throwable
     */
    public function receive(ReceiveSalesCreditRequest $request): Response
    {
        $inputs = $request->validated();

        DB::transaction(function () use ($inputs) {
            $salesCredit = SalesCredit::with([])->findOrFail($inputs['sales_credit_id']);

            // add a new sales credit return
            $salesCreditReturn = new SalesCreditReturn($inputs);
            $salesCredit->salesCreditReturns()->save($salesCreditReturn);

            // add its lines and add to inventory
            $salesCreditReturn->setReturnLines($inputs['return_lines']);

            // mark sales credit as returned
            $salesCredit->returned($salesCreditReturn->received_at);

            $this->response->addData(SalesCreditResource::make($salesCredit->load($this->getRequiredRelations())));
        });

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'sales credit return']));
    }

    /**
     * Add payment to Sales Credit.
     *
     *
     * @throws Throwable
     */
    public function paid(PaidSalesCreditRequest $request): Response
    {
        $inputs = $request->validated();
        DB::transaction(function () use ($inputs) {
            $salesCredit = SalesCredit::with([])->findOrFail($inputs['sales_credit_id']);

            // add a new payment
            // Make amount negative
            if (isset($inputs['amount'])) {
                $inputs['amount'] = abs($inputs['amount']);
            }
            $payment = new Payment($inputs);
            $salesCredit->payments()->save($payment);

            // mark sales credit as returned
            $salesCredit->paid($payment->payment_date);

            $this->response->addData(SalesCreditResource::make($salesCredit->load($this->getRequiredRelations())));
        });

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'payment']));
    }

    /**
     * Archive the sales credit.
     */
    public function archive(SalesCredit $salesCredit): Response
    {
        if ($salesCredit->archive()) {
            return $this->response->setMessage(__('messages.success.archive', [
                'resource' => 'sales credit',
                'id' => $salesCredit->sales_credit_number,
            ]));
        }

        return $this->response->addWarning(__('messages.failed.already_archive', [
            'resource' => 'sale credit',
            'id' => $salesCredit->sales_credit_number,
        ]), 'SalesCredit'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $salesCredit->id]);
    }

    /**
     * Unarchived.
     */
    public function unarchived(SalesCredit $salesCredit): Response
    {
        if ($salesCredit->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'sales credit',
                    'id' => $salesCredit->sales_credit_number,
                ]))
                ->addData(SalesCreditResource::make($salesCredit));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'sales credit',
                'id' => $salesCredit->sales_credit_number,
            ]), 'SalesCredit'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $salesCredit->id])
            ->addData(SalesCreditResource::make($salesCredit));
    }

    /**
     * Remove sales credit.
     *
     *
     * @throws Throwable
     */
    public function destroy(SalesCredit $salesCredit): Response
    {
        $salesCredit->delete();

        return $this->response->setMessage(__(
            'messages.success.delete',
            [
                'resource' => 'sales credit',
                'id' => $salesCredit->sales_credit_number,
            ]
        ));
    }

    /**
     * Delete Sales credit return line.
     *
     *
     * @throws Exception
     */
    public function destroyReturnLine(SalesCreditReturnLine $salesCreditReturnLine): Response
    {
        // if the sales credit return has just this line, remove sales credit return with this line
        if ($salesCreditReturnLine->salesCreditReturn->salesCreditReturnLines()->count() == 1) {
            $salesCreditReturnLine->salesCreditReturn->delete();
        } else {
            // if sales credit return has another lines, just delete this line
            $salesCreditReturnLine->delete();
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'return line',
            'id' => $salesCreditReturnLine->id,
        ]));
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:sales_credits,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $salesCredits = SalesCredit::with([])->whereIn('id', $ids)->select('id', 'sales_credit_number')->get();
        foreach ($salesCredits as $key => $salesCredit) {
            $isUsed = $salesCredit->isUsed();

            $result[$key] = $salesCredit->only('id', 'sales_credit_number');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * bulk delete using request filters or body ids array.
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * bulk archive using request filters or body ids array.
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return SalesCredit::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return SalesCreditResource::class;
    }

    /**
     * Get Required Relations.
     */
    protected function getRequiredRelations(bool $view = true): array
    {
        $baseRelations = DataTableConfiguration::getRequiredRelations(SalesCredit::class);

        $relationsForView = [
            'salesCreditLines.product.primaryImage',
            'salesCreditLines.salesCreditReturnLines',
            'salesCreditLines.salesCreditReturnLines.salesCreditReturn', // for received_at
            'salesCreditLines.salesCreditReturnLines.returnReason',
            'payments',
            'allocations',
        ];

        return $view ? array_merge($baseRelations, $relationsForView) : $baseRelations;
    }

    /**
     * Add note to sales credit.
     */
    public function addNote(Request $request, SalesCredit $salesCredit): Response
    {
        $request->validate(['note' => 'required']);

        $note = $salesCredit->notes()->create($request->only('note'));
        $note->load('user');

        return $this->response->addData($note)
            ->setMessage(__('messages.success.create', ['resource' => 'note']));
    }

    /**
     * View sales credit notes.
     */
    public function notes(SalesCredit $salesCredit): JsonResponse
    {
        return $this->response->setData($salesCredit->notes()->with('user')->orderByDesc('created_at')->paginate()->toArray());
    }

    /**
     * Deletes a sales credit note.
     */
    public function deleteNote(SalesCredit $salesCredit, $noteId): Response
    {
        $note = $salesCredit->notes()->find(e($noteId));
        if ($note) {
            $note->delete();
        }

        return $this->response->setMessage(__('messages.success.delete', ['resource' => 'note', 'id' => $noteId]));
    }
}
