<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Resources\ReportingDailyFinancialResource;
use App\Jobs\GenerateRecalculateDailyFinancialsJob;
use App\Models\ReportingDailyFinancial;
use App\Response;
use App\Services\FinancialManagement\DailyFinancialManager;
use Illuminate\Database\QueryException;
use Illuminate\Http\Request;

class ReportingDailyFinancialController extends Controller
{
    use DataTable;

    protected function getModel()
    {
        return ReportingDailyFinancial::class;
    }

    protected function getResource()
    {
        return ReportingDailyFinancialResource::class;
    }

    public function recalculateDailyFinancials(Request $request): Response
    {
        $this->validate($request, [
            'ids' => 'sometimes|array|min:1',
            'ids.*' => 'integer|exists:reporting_daily_financials,id',
            'filters' => 'sometimes|string',
        ]);

        if ($request->has('filters')) {
            try {
                $request->merge([
                        'ids' => $this->getIdsFromFilters(
                            modelClass: ReportingDailyFinancial::class,
                            request: $request,
                        )
                    ]
                );
            } catch (QueryException) {
                return $this->response->error(400)->addError('The filters are too complex to support select all results currently.  Please select one page at a time.', Response::HTTP_BAD_REQUEST, 'too_complex');
            }

            dispatch(new GenerateRecalculateDailyFinancialsJob(ReportingDailyFinancial::whereIn('id', $request->input('ids'))->get()));
            return $this->response->setMessage('Recalculations processing in the background');
        }

        (new DailyFinancialManager())->recalculateDailyFinancials(ReportingDailyFinancial::whereIn('id', $request->input('ids'))->get());

        return $this->response->setMessage('Recalculation successful');
    }
}
