<?php

namespace App\Http\Controllers;

use App\Data\FinancialSummaryQueryData;
use App\Managers\ReportManager;
use App\Repositories\ReportingRepository;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class ReportingController extends Controller
{
    protected ReportingRepository $reportingRepository;

    public function __construct()
    {
        parent::__construct();
        $this->reportingRepository = app(ReportingRepository::class);
    }

    /**
     * @throws Exception
     */
    public function purchaseSummaryByPeriod(Request $request)
    {
        $request->validate([
            'period' => 'required|string',
            'trailing_days' => 'nullable|integer',
        ]);

        return JsonResource::collection(
            $this->reportingRepository->getPurchaseSummaryByPeriod($request->input('period'), $request->input('trailing_days'))
        )
            ->additional([
                'status' => __('messages.status_success'),
            ]);
    }

    public function agingInventory(): Response
    {
        return $this->response->addData(app(ReportManager::class)->agingInventory());
    }

    public function valuation()
    {
        return $this->response->addData(app(ReportManager::class)->valuationSummary());
    }

    public function valuationByProduct()
    {
        return $this->response->addData(app(ReportManager::class)->valuationSummaryByProduct());
    }

    private function getFinancialsBy(string $method, FinancialSummaryQueryData $data)
    {
        return $this->response->addData(app(ReportManager::class)->$method($data));
    }

    public function financialsByBrand(FinancialSummaryQueryData $data)
    {
        return $this->getFinancialsBy('getFinancialsByBrand', $data);
    }

    public function financialsBySupplier(FinancialSummaryQueryData $data)
    {
        return $this->getFinancialsBy('getFinancialsBySupplier', $data);
    }

    public function financialsBySku(FinancialSummaryQueryData $data)
    {
        return $this->getFinancialsBy('getFinancialsBySku', $data);
    }

    public function financialsByProductType(FinancialSummaryQueryData $data)
    {
        return $this->getFinancialsBy('getFinancialsByProductType', $data);
    }

    public function financialsBySalesChannel(FinancialSummaryQueryData $data)
    {
        return $this->getFinancialsBy('getFinancialsBySalesChannel', $data);
    }

    public function getSalesChannelCoverage(Request $request): Response
    {
        $request->validate([
            'byIntegration' => 'required|boolean',
            'showNumberOfCoverages' => 'required|boolean',
            'ids' => 'required|array',
        ]);


        $data = app(ReportManager::class)->getSalesChannelCoverage($request->input('byIntegration'), $request->input('showNumberOfCoverages'), $request->input('ids'));
        return $this->response->addData($data);
    }
}
