<?php

namespace App\Http\Controllers;

use App\Exceptions\InsufficientStockException;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\NoWarehouseException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Http\Requests\ReceiptPurchaseOrderRequest;
use App\Http\Requests\StorePurchaseOrderShipment;
use App\Http\Resources\PurchaseOrderReceiptResource;
use App\Http\Resources\PurchaseOrderShipmentResource;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Notifications\MonitoringMessage;
use App\Response;
use App\Services\PurchaseOrder\ShipmentManager;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Throwable;

class PurchaseOrderShipmentController extends Controller
{
    private ShipmentManager $manager;

    /**
     * PurchaseOrderShipmentController constructor.
     */
    public function __construct(ShipmentManager $manager)
    {
        parent::__construct();
        $this->manager = $manager;
    }

    /**
     * Create a new PO Shipment.
     *
     *
     * @throws Throwable
     */
    public function store(StorePurchaseOrderShipment $request): Response
    {
        try {
            $shipment = $this->manager->createShipment($request->validated());

            return $this->response->addData(PurchaseOrderShipmentResource::make($shipment))
                ->setMessage(__('messages.success.create', ['resource' => 'purchase order shipment']));
        } catch (NotOpenPurchaseOrderException $e) {
            return $this->response->error()->addError(__('', [
                'id' => $e->purchaseOrder->id,
                'resource' => 'purchase order',
            ]), 'PurchaseOrder', 'id', ['id' => $e->purchaseOrder->id]);
        } catch (\Exception $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR)
                ->setMessage($e->getMessage());
        }
    }

    /**
     * Update PO Shipment.
     */
    public function update(StorePurchaseOrderShipment $request, PurchaseOrderShipment $purchaseOrderShipment): Response
    {
        $purchaseOrderShipment->update($request->validated());

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => 'purchase order shipment',
                'id' => $purchaseOrderShipment->id,
            ]))
            ->addData(PurchaseOrderShipmentResource::make($purchaseOrderShipment));
    }

    /**
     * Receive Purchase Order Shipment.
     *
     *
     * @throws Throwable
     */
    public function receive(ReceiptPurchaseOrderRequest $request): Response
    {
        try {
            /*
             * Temporary notification to capture purchase receipt payloads
             */
            //Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage("Purchase receipt payload: " . $request->getContent()));
            $resource = $this->manager->receiveShipment($request->validated());

            if ($resource instanceof PurchaseOrderShipmentReceipt) {
                return $this->response->setMessage(__('messages.success.create', ['resource' => 'purchase order shipment receipt']))
                    ->addData(PurchaseOrderReceiptResource::make($resource));
            }

            return $this->response->setMessage(__('messages.success.create', ['resource' => 'purchase order shipment receipt']));
        } catch (NotOpenPurchaseOrderException $e) {
            return $this->response->error()->addError(__('messages.purchase_order.is_not_open', [
                'id' => $e->purchaseOrder->id,
                'resource' => 'purchase order',
            ]), 'PurchaseOrder', 'id', ['id' => $e->purchaseOrder->id]);
        } catch (NoWarehouseException $e) {
            return $this->response->error()->addError(__('custom_validation.required', [
                'attribute' => 'warehouse_id',
                'index' => '',
            ]), 'WarehouseIdRequired', 'warehouse_id');
        } catch (ReceivePurchaseOrderLineException $e) {
            return $this->response->error()->addError(__('messages.purchase_order.can_not_receive_lines'), 'PurchaseOrder', 'id', ['id' => @$request->validated()['purchase_order_id']]);
        }
    }

    /**
     * Gets the shipment receipts for a purchase order.
     */
    public function receipts($purchaseOrderId): Response
    {
        $purchaseOrder = PurchaseOrder::with([])->findOrFail(e($purchaseOrderId));
        $receipts = PurchaseOrderShipmentReceipt::with([
            'purchaseOrderShipment.purchaseOrder',
            'purchaseOrderShipmentReceiptLines',
        ])
            ->whereHas('purchaseOrderShipment.purchaseOrder', function (Builder $builder) use ($purchaseOrder) {
                return $builder->where('id', $purchaseOrder->id);
            });

        return $this->response->addData(PurchaseOrderReceiptResource::collection($receipts->get()));
    }

    public function getReceipt($purchaseOrderId, $receiptId): Response
    {
        $purchaseOrder = PurchaseOrder::with([])->findOrFail(e($purchaseOrderId));
        $receipt = PurchaseOrderShipmentReceipt::with([
            'purchaseOrderShipment.purchaseOrder',
            'purchaseOrderShipmentReceiptLines',
        ])
            ->whereHas('purchaseOrderShipment.purchaseOrder', function (Builder $builder) use ($purchaseOrder) {
                return $builder->where('id', $purchaseOrder->id);
            })->findOrFail(e($receiptId));

        return $this->response->addData(PurchaseOrderReceiptResource::make($receipt));
    }

    public function updateReceipt(ReceiptPurchaseOrderRequest $request, $receiptId): Response
    {
        /** @var PurchaseOrderShipmentReceipt $receipt */
        $receipt = PurchaseOrderShipmentReceipt::with(['purchaseOrderShipmentReceiptLines'])->findOrFail(e($receiptId));

        try {
            $resource = $this->manager->updateShipmentReceipt($receipt, $request->validated());

            return $this->response->setMessage(__('messages.success.update', ['resource' => 'purchase order shipment receipt']))
                ->addData(PurchaseOrderReceiptResource::make($resource));
        } catch (InsufficientStockException $e) {
            // Some product doesn't have enough fifo layer
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.product.initial_stock_insufficient_fifo_layer', [
                    'extra' => '',
                    'sku' => $e->productSku,
                    'product_id' => $e->productId,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        } catch (NotOpenPurchaseOrderException $e) {
            return $this->response->error()->addError(__('messages.purchase_order.is_not_open', [
                'id' => $e->purchaseOrder->id,
                'resource' => 'purchase order',
            ]), 'PurchaseOrder', 'id', ['id' => $e->purchaseOrder->id]);
        } catch (\InvalidArgumentException $e) {
            return $this->response->error()->addError($e->getMessage(), 'PurchaseOrderReceipt', 'line');
        }
    }

    public function deleteReceipt($receiptId): Response
    {
        /** @var PurchaseOrderShipmentReceipt $receipt */
        $receipt = PurchaseOrderShipmentReceipt::with(['purchaseOrderShipmentReceiptLines'])->findOrFail(e($receiptId));

        try {
            $this->manager->deleteReceipt($receipt);

            return $this->response->setMessage(__('messages.success.delete', ['resource' => 'purchase order shipment receipt']));
        } catch (InsufficientStockException $e) {
            return $this->response->error()->addError(__('messages.purchase_order.shipment.used_receipt_fifo_layer', [
                'sku' => $e->productSku,
                'resource' => 'Product',
            ]), 'Product', 'sku', ['sku' => $e->productSku]);
        }
    }

    public function deleteReceiptLines(Request $request, $receiptId)
    {
        request()->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:purchase_order_shipment_receipt_lines,id',
        ]);

        /** @var PurchaseOrderShipmentReceipt $receipt */
        $receipt = PurchaseOrderShipmentReceipt::with(['purchaseOrderShipmentReceiptLines'])->findOrFail(e($receiptId));

        try {
            $this->manager->deleteReceipt($receipt, $request->all());

            return $this->response->setMessage(__('messages.success.delete', ['resource' => 'purchase order shipment receipt']));
        } catch (InsufficientStockException $e) {
            return $this->response->error()->addError(__('messages.purchase_order.shipment.used_receipt_fifo_layer', [
                'sku' => $e->productSku,
                'resource' => 'Product',
            ]), 'Product', 'sku', ['sku' => $e->productSku]);
        }
    }
}
