<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\CannotDeleteReceivedPurchaseOrderLineException;
use App\Exceptions\DuplicateEntryException;
use App\Exceptions\NegativeInventoryFulfilledSalesOrderLinesException;
use App\Exceptions\ReceivingDiscrepanciesMissingNominalCodeMappingException;
use App\Exceptions\ReceivingDiscrepancyAlreadyExistsException;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\ImportCSVFileRequest;
use App\Http\Requests\StorePurchaseOrder;
use App\Http\Resources\AccountingTransactionResource;
use App\Http\Resources\PurchaseOrderResource;
use App\Http\Resources\SalesOrderFulfillmentResource;
use App\Jobs\GeneratePurchaseOrderInvoice;
use App\Models\AccountingTransaction;
use App\Models\PurchaseOrder;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Repositories\InventoryForecastRepository;
use App\Response;
use App\Services\Accounting\AccountingTransactionManager;
use App\Services\PurchaseOrder\BulkApprovePurchaseOrderService;
use App\Services\PurchaseOrder\BulkSubmitPurchaseOrderService;
use App\Services\PurchaseOrder\PurchaseOrderManager;
use App\Services\PurchaseOrder\PurchaseOrderValidator;
use Exception;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Managers\AmazonInboundManager;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Throwable;

class PurchaseOrderController extends Controller
{
    use bulkOperation, DataTable, ImportsData;

    protected $model_path = PurchaseOrder::class;

    /**
     * @var InventoryForecastRepository
     */
    protected $forecastCache;

    public PurchaseOrderManager $purchaseOrderManager;

    /**
     * PurchaseOrderController constructor.
     */
    public function __construct(InventoryForecastRepository $forecastCache, PurchaseOrderManager $purchaseOrderManager)
    {
        parent::__construct();
        $this->forecastCache = $forecastCache;
        $this->purchaseOrderManager = $purchaseOrderManager;
    }

    /**
     * Get a purchase order by id.
     */
    public function show(int $id): PurchaseOrderResource
    {
        // Get purchase order by id with relations Or fail 404 not found.
        $purchaseOrders = PurchaseOrder::with([
            'purchaseOrderLines.product.totalInventory',
            'purchaseOrderLines.product.suppliersInventory',
            'purchaseOrderLines.product.productInventory',
            'purchaseOrderLines.coveredBackorderQueues',
            'purchaseOrderLines.purchaseOrderShipmentReceiptLines',
            'purchaseOrderLines.adjustments.product',
            'inboundNewShipmentRelation',
            'inboundShipmentRelation',
            'adjustments.product',
            'notes',
            'destinationWarehouse'
        ])->addRelations()->findOrFail($id);

        return PurchaseOrderResource::make($purchaseOrders);
    }

    /**
     * Store new purchase order and fire events.
     *
     *
     * @throws Throwable
     */
    public function store(StorePurchaseOrder $request): Response
    {
        /** @see SKU-4423 */
        set_time_limit(60 * 10);

        // All inputs as array
        $inputs = $request->validated();

        // Transaction to rollback if any exception occur.
        return DB::transaction(function () use ($inputs) {
            // Save purchase order and lines.
            try {
                $purchaseOrder = $this->savePurchaseOrder($inputs, 2);
            } catch (DuplicateEntryException $exception) {
                return $this->response->error(Response::HTTP_BAD_REQUEST)
                    ->addError($exception->getMessage(), $exception->getResponseCode(), 'purchase_order_number');
            }

            $lines = $inputs['purchase_order_lines'];
            if (empty($lines)) {
                $lines = $inputs['forecasting_products'] ?? null;
            }

            $purchaseOrder->setPurchaseOrderLines($lines);

            if (isset($inputs['tags'])) {
                // If the tags input parameter is provided, then set
                // the provided tags (Any tags which are not present in the payload will be removed)
                // If the tags input parameter is not provided, then leave the current tags as they are
                $purchaseOrder->syncTags($inputs['tags']);
            }

            // Update forecast cache if necessary
            if (! empty($inputs['forecasting_products'])) {
                $this->forecastCache->setPurchaseOrderLines($inputs['forecasting_products'], $purchaseOrder->purchaseOrderLines);
            }

            // approve purchase order
            if (($inputs['approval_status'] ?? null) == PurchaseOrderValidator::APPROVAL_STATUS_APPROVED) {
                if (is_array($res = $purchaseOrder->approve())) {
                    $this->response->addWarning(...$res);
                }
            }

            if ($purchaseOrder->destinationWarehouse && $purchaseOrder->destinationWarehouse->type == Warehouse::TYPE_AMAZON_FBA) {
                $integrationInstance = AmazonIntegrationInstance::find($purchaseOrder->destinationWarehouse->integrationInstance->id);
                (new AmazonInboundManager($integrationInstance))->linkPurchaseOrder($purchaseOrder, $inputs['shipment_id']);
            }

            $purchaseOrder->load(DataTableConfiguration::getRequiredRelations(PurchaseOrder::class));
            $purchaseOrder->loadMissing('purchaseOrderLines.product.suppliersInventory');
            $purchaseOrder->loadMissing('purchaseOrderLines.product.totalInventory');
            $this->response->addData(PurchaseOrderResource::make($purchaseOrder));

            return $this->response->setMessage(__('messages.success.create', ['resource' => 'purchase order']));
        });
    }

    /**
     * Save purchase order data and attempt to solve duplicate entry exception
     * every attempt will try to get the next purchase order number
     *
     * TODO: Purchase orders can have a lot of lines so need a bulk method here
     *
     * @throws DuplicateEntryException|QueryException
     */
    private function savePurchaseOrder(array $inputs, int $attempts = 5): PurchaseOrder
    {
        $purchaseOrder = new PurchaseOrder($inputs);
        try {
            $purchaseOrder->save();
        } catch (QueryException $queryException) {
            // SQLSTATE[23000]: Identify constraint violation: 1062 Duplicate entry
            $sqlState = $queryException->errorInfo[0] ?? 0;
            if ($sqlState == 23000) {
                if (! isset($inputs['purchase_order_number']) && $attempts) {
                    return $this->savePurchaseOrder($inputs, --$attempts);
                }

                throw new DuplicateEntryException($purchaseOrder, 'purchase_order_number', $queryException);
            }

            throw $queryException;
        }

        return $purchaseOrder;
    }

    /**
     * Update Purchase Order and fire event if approved.
     *
     * TODO: Purchase orders can have a lot of lines so need a bulk method here
     *
     *
     * @throws Throwable
     */
    public function update(StorePurchaseOrder $request, PurchaseOrder $purchaseOrder): Response
    {

        // All inputs as array
        $inputs = $request->validated();

        if ($purchaseOrder->inboundShipmentRelation &&
            $this->purchaseOrderManager->changingProductLines($purchaseOrder, @$inputs['purchase_order_lines'] ?? [])) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.purchase_order.purchase_order_is_locked'));
        }

        DB::beginTransaction();
        try {
            if (@$inputs['po_number']) {
                $purchaseOrder->purchase_order_number = $inputs['po_number'];
            }

            if (isset($inputs['supplier_id'])) {
                $supplier = Supplier::findOrFail($inputs['supplier_id']);
                $warehouse = Warehouse::where('supplier_id', $supplier->id)->where('id', $supplier->default_warehouse_id)->firstOrFail();
                $inputs['supplier_warehouse_id'] = $warehouse->id;
            }

            $purchaseOrder->update($inputs);

            if (isset($inputs['tags'])) {
                // If the tags input parameter is provided, then set
                // the provided tags (Any tags which are not present in the payload will be removed)
                // If the tags input parameter is not provided, then leave the current tags as they are
                $purchaseOrder->syncTags($inputs['tags']);
            }

            // update lines
            $purchaseOrder->setPurchaseOrderLines(
                array_key_exists('purchase_order_lines', $inputs) ? $inputs['purchase_order_lines'] : false
            );

            // approve purchase order
            if (($inputs['approval_status'] ?? null) == StorePurchaseOrder::APPROVAL_STATUS_APPROVED) {
                if (is_array($res = $purchaseOrder->approve())) {
                    $this->response->addWarning(...$res);
                }
            }

            // revert to draft purchase order
            if (($inputs['order_status'] ?? null) == PurchaseOrder::STATUS_DRAFT) {
                $purchaseOrder->revertToDraft();
            }

            // Set invoice status
            $purchaseOrder->invoiced();

            //      if ( ! isset( $inputs['approval_status'] ) || $inputs['approval_status'] !== StorePurchaseOrder::APPROVAL_STATUS_APPROVED )
            //      {
            //        // Notify supplier about the update.
            //        $purchaseOrder->supplier->notify( new PurchaseOrderUpdatedNotification( $purchaseOrder ) );
            //      }

            $purchaseOrder->load(DataTableConfiguration::getRequiredRelations(PurchaseOrder::class));
            $purchaseOrder->loadMissing('purchaseOrderLines.product.productInventory');
            $purchaseOrder->loadMissing('purchaseOrderLines.product.suppliersInventory');
            $purchaseOrder->loadMissing('purchaseOrderLines.product.totalInventory');
            $this->response->addData(PurchaseOrderResource::make($purchaseOrder));

            DB::commit();
        } catch (CannotDeleteReceivedPurchaseOrderLineException) {
            DB::rollBack();

            return $this->response->setMessage('Cannot delete a purchase order line that has been received')
                ->setStatus(422);
        } catch (Throwable $exception) {
            try {
                DB::rollBack();
            } catch (Throwable $rollbackException) {
                //                throw new QueryException($rollbackException->getMessage(), [], $exception);
            }

            throw $exception;
        }

        return $this->response->setMessage(__('messages.success.update', [
            'resource' => 'purchase order',
            'id' => $purchaseOrder->purchase_order_number,
        ]));
    }

    /**
     * submit and open draft purchase order.
     */
    public function submit(PurchaseOrder $purchaseOrder): Response
    {
        if (is_array($res = $purchaseOrder->submit())) {
            return $this->response->error()
                ->addError(...$res)
                ->setMessage(__('messages.purchase_order.not_submitted_to_supplier', [
                    'resource' => 'purchase order',
                    'id' => $purchaseOrder->purchase_order_number,
                ]));
        }

        $this->response->setMessage(
            __('messages.purchase_order.submit_to_supplier', ['id' => $purchaseOrder->purchase_order_number])
        );

        return $this->response->addData(PurchaseOrderResource::make($purchaseOrder));
    }

    public function downloadInvoiceCSV(PurchaseOrder $purchaseOrder): BinaryFileResponse
    {
        return response()->download($purchaseOrder->linesToCSV());
    }

    public function downloadInvoicePDF(PurchaseOrder $purchaseOrder): BinaryFileResponse
    {
        return response()->download($purchaseOrder->exportToPDF());
    }

    public function picklist(PurchaseOrder $purchaseOrder): BinaryFileResponse
    {
        return response()->download($purchaseOrder->exportToPDF('picklist'));
    }

    /**
     * Duplicate sales order by id.
     */
    public function duplicate($purchaseOrderId): PurchaseOrderResource
    {
        $replicatedPurchaseOrder = $this->purchaseOrderManager->duplicateOrder($purchaseOrderId);
        $this->response->setMessage(__('messages.sales_order.duplicate_success'));
        return $this->show($replicatedPurchaseOrder->id);
    }

    /**
     * Bulk approve purchase orders.
     */
    public function bulkApprove(Request $request): Response
    {
        $request->validate(['ids' => 'required_without:filter|array', 'filters' => 'required_without:ids']);

        $bulkApproveService = new BulkApprovePurchaseOrderService($request);

        $response = $bulkApproveService->approve();
        $approveCount = count(Arr::flatten($response, 1));
        if ($approveCount == 0) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.bulk_po_approve_empty'));
        }
        // all purchase orders failed to approve
        if ($approveCount == $bulkApproveService->getErrorsCount()) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.bulk_po_submit'))->setErrors($response);
        }
        // some purchase orders failed to approve
        if ($bulkApproveService->getErrorsCount()) {
            return $this->response->warning()
                ->setMessage(__('messages.purchase_order.not_all_submitted', [
                    'success' => ($approveCount - $bulkApproveService->getErrorsCount()),
                    'total' => $approveCount,
                ]))->setWarnings($response);
        }

        // all purchase orders approved successfully
        return $this->response->setMessage(__('messages.success.bulk_po_approve'))->addData($response);
    }

    /**
     * Bulk submit purchase orders.
     */
    public function bulkSubmit(Request $request): Response
    {
        $request->validate(['ids' => 'required_without:filter|array', 'filters' => 'required_without:ids']);

        $bulkSubmitService = new BulkSubmitPurchaseOrderService($request);

        $response = $bulkSubmitService->submit();
        $submitCount = count(Arr::flatten($response, 1));
        if ($submitCount == 0) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.bulk_po_submit_empty'));
        }
        // all purchase orders failed to submit
        if ($submitCount == $bulkSubmitService->getErrorsCount()) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.bulk_po_submit'))->setErrors($response);
        }
        // some purchase orders failed to submit
        if ($bulkSubmitService->getErrorsCount()) {
            return $this->response->warning()
                ->setMessage(__('messages.purchase_order.not_all_so_approved', [
                    'success' => ($submitCount - $bulkSubmitService->getErrorsCount()),
                    'total' => $submitCount,
                ]))->setWarnings($response);
        }

        // all purchase orders submitted successfully
        return $this->response->setMessage(__('messages.success.bulk_po_submit'))->addData($response);
    }

    /**
     * Import Purchase Orders from CSV file.
     *
     * open orders should be imported and waiting for approvements
     *
     * @param  ImportCSVFileRequest  $request
     * @return JsonResponse
     *
     * @throws Throwable
     */
    //  public function importCSV( ImportCSVFileRequest $request )
    //  {
    //    $purchaseOrders = [];
    //    $successCount   = 0;
    //
    //    foreach ( $request->getRows() as $index => $row )
    //    {
    //      // apply validation rules on the row
    //      $validator = Validators::purchaseOrderLineFromCSV( $row );
    //      if ( $validator->fails() )
    //      {
    //        $this->response->addWarningsFromValidator( $validator, "purchase_orders.{$index}" );
    //        continue;
    //      }
    //
    //      // get or create a new sales order
    //      // we use this check to prevent make useless query
    //      $purchaseOrder = empty( $row['purchase_order_id'] ) ? new PurchaseOrder() : PurchaseOrder::with( [] )->findOrFail( $row['purchase_order_id'] );
    //      $purchaseOrder->fill( $row );
    //
    //      // skip the existing sales order if override data is false
    //      if ( $purchaseOrder->exists && ! $request->overrideData() )
    //      {
    //        continue;
    //      }
    //
    //      $purchaseOrder->save();
    //
    //      // add to draft sales order to adding/updating lines and approve it if order status changed to Open
    //      if ( isset( $purchaseOrders[ $purchaseOrder->id ] ) )
    //      {
    //        $purchaseOrders[ $purchaseOrder->id ]['lines'][] = $request->getElementsByPrefix( $row, 'order_line_' );
    //      } else
    //      {
    //        $purchaseOrders[ $purchaseOrder->id ] = [
    //          'index' => $index,
    //          'id'    => $purchaseOrder->id,
    //          'lines' => [ $request->getElementsByPrefix( $row, 'order_line_' ) ],
    //        ];
    //      }
    //
    //      $successCount ++;
    //    }
    //
    //    // add/update purchase order lines
    //    // we add purchase order lines after fetching whole file to sync purchase order lines (create/update/delete)
    //    foreach ( array_chunk( $purchaseOrders, 20 ) as $orders )
    //    {
    //      $orderIds = array_column( $orders, 'id' );
    //      /** @var PurchaseOrder $purchaseOrder */
    //      foreach ( PurchaseOrder::with( [] )->findMany( $orderIds ) as $purchaseOrder )
    //      {
    //        $order = collect( $orders )->firstWhere( 'id', $purchaseOrder->id );
    //
    //        // syncing sales order lines
    //        $purchaseOrder->setPurchaseOrderLines( $order['lines'] );
    //      }
    //    }
    //
    //    // at least one purchase order added successfully
    //    if ( $successCount )
    //    {
    //      return $this->response->setMessage( __( 'messages.success.import', [ 'resource' => 'purchase orders' ] ) );
    //    } else
    //    {
    //      return $this->response->setMessage( __( 'messages.failed.csv_all_lines_incorrect' ) );
    //    }
    //  }

    /**
     * bulk delete using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    public function destroy(PurchaseOrder $purchaseOrder)
    {
        try {
            $reasons = $purchaseOrder->delete();
        } catch (NegativeInventoryFulfilledSalesOrderLinesException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage('This purchase order cannot be deleted because it is received and the receipts were used.  And there are no other positive inventory events to offset the negative inventory events.  You must either delete the sales order usages or create other positive inventory events in the place of this purchase order');
        }

        // check if the nominalCode is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['purchase_order_id' => $purchaseOrder->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'purchase order',
                    'id' => $purchaseOrder->purchase_order_number,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'purchase order',
            'id' => $purchaseOrder->purchase_order_number,
        ]));
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:purchase_orders,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $purchaseOrders = PurchaseOrder::with([])->whereIn('id', $ids)->select('id', 'purchase_order_number')->get();
        foreach ($purchaseOrders as $index => $purchaseOrder) {
            $isDeletable = $purchaseOrder->isDeletable();

            $result[$index] = $purchaseOrder->only('id', 'purchase_order_number');
            $result[$index]['deletable'] = $isDeletable;
            $result[$index]['reason'] = $isDeletable ? null : ['usedFifoLayers' => __('messages.purchase_order.used_fifo_layers_in_lines', ['id' => $purchaseOrder->purchase_order_number])];
        }

        return $this->response->addData($result);
    }

    /**
     * Get dropship shipments for the dropship purchase order.
     */
    public function dropshipShipments(PurchaseOrder $purchaseOrder): Response
    {
        // non-dropship purchase order
        if (empty($purchaseOrder->sales_order_id)) {
            return $this->response->addData([]);
        }

        $purchaseOrder->salesOrder->load([
            'salesOrderFulfillments' => function (HasMany $builder) use ($purchaseOrder) {
                // only fulfillments that belong to supplier warehouse of the purchase order
                $builder->where('warehouse_id', $purchaseOrder->supplier_warehouse_id);
            },
            'salesOrderFulfillments.salesOrderFulfillmentLines',
            'salesOrderFulfillments.requestedShippingMethod.shippingCarrier',
        ]);

        return $this->response->addData(SalesOrderFulfillmentResource::collection($purchaseOrder->salesOrder->salesOrderFulfillments));
    }

    public function previewInvoice(PurchaseOrder $purchaseOrder)
    {
        $path = dispatch_sync(new GeneratePurchaseOrderInvoice($purchaseOrder));
        $path = Storage::disk('reports')->url($path);

        return $this->response->addData(compact('path'));
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return PurchaseOrder::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return PurchaseOrderResource::class;
    }

    public function notes(PurchaseOrder $purchaseOrder)
    {
        return $this->response->setData($purchaseOrder->notes()->with('user')->orderByDesc('created_at')->paginate()->toArray());
    }

    public function addNote(Request $request, PurchaseOrder $purchaseOrder)
    {
        $request->validate(['note' => 'required']);

        $note = $purchaseOrder->notes()->create($request->only('note'));
        $note->load('user');

        return $this->response->addData($note)
            ->setMessage(__('messages.success.create', ['resource' => 'note']));
    }

    public function deleteNote(PurchaseOrder $purchaseOrder, $noteId)
    {
        $note = $purchaseOrder->notes()->find(e($noteId));
        if ($note) {
            $note->delete();
        }

        return $this->response->setMessage(__('messages.success.delete', ['resource' => 'note', 'id' => $noteId]));
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function createReceivingDiscrepancy(PurchaseOrder $purchaseOrder): Response
    {
        try {
            $accountingTransaction = app(AccountingTransactionManager::class)->createReceivingDiscrepancyFromPurchaseOrder($purchaseOrder);
        } catch (ReceivingDiscrepanciesMissingNominalCodeMappingException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError($e->getMessage(), 'ReceivingDiscrepancyMissingNominalCodeMapping', 'nominal_code_id');
        } catch (ReceivingDiscrepancyAlreadyExistsException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError($e->getMessage(), 'ReceivingDiscrepancyAlreadyExists', 'purchase_order_id');
        }

        return $this->response->addData(AccountingTransactionResource::make($accountingTransaction));
    }
}
