<?php

namespace App\Http\Controllers;

use App\Events\CustomPurchaseInvoiceCreated;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\StoreCustomPurchaseInvoice;
use App\Http\Requests\StorePurchaseInvoice;
use App\Http\Resources\PurchaseInvoiceResource;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceLine;
use App\Models\PurchaseOrder;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Throwable;

class PurchaseInvoiceController extends Controller
{
    use BulkOperation;

    protected $model_path = PurchaseInvoice::class;

    public function index($purchaseOrderId): Response
    {
        $purchaseOrder = PurchaseOrder::with([
            'purchaseInvoices',
            'purchaseInvoices.purchaseInvoiceLines',
        ])->findOrFail(e($purchaseOrderId));

        return $this->response->addData(PurchaseInvoiceResource::collection($purchaseOrder->purchaseInvoices));
    }

    /**
     * Create new purchase invoice.
     *
     *
     * @throws Throwable
     */
    public function store(StorePurchaseInvoice $request): Response
    {
        $inputs = $request->validated();

        DB::beginTransaction();
        try {
            // Get purchase order
            $purchaseOrder = PurchaseOrder::with([])->find($inputs['purchase_order_id']);

            // Save Purchase Invoice
            $purchaseInvoice = new PurchaseInvoice($inputs);
            $purchaseInvoice->supplier_id = $purchaseOrder->supplier_id;
            $purchaseOrder->purchaseInvoices()->save($purchaseInvoice);

            // Aggregate invoice lines
            $inputs['purchase_invoice_lines'] = PurchaseInvoice::aggregateInvoiceLines($inputs['purchase_invoice_lines']);

            $purchaseInvoiceLines = $inputs['purchase_invoice_lines'];

            // update/create purchase order lines(without delete)
            $purchaseInvoiceLines = $purchaseOrder->setPurchaseOrderLines($purchaseInvoiceLines, false);

            $purchaseInvoiceLines = collect($purchaseInvoiceLines)->map(function ($line) {
                return Arr::only($line, ['quantity_invoiced', 'purchase_order_line_id']);
            })->toArray();

            // save purchase invoice lines
            $purchaseInvoice->purchaseInvoiceLines()->createMany($purchaseInvoiceLines);

            $purchaseOrder->invoiced($purchaseInvoice->purchase_invoice_date);

            $this->response->addData(PurchaseInvoiceResource::make($purchaseInvoice));

            DB::commit();
        } catch (Throwable $exception) {
            try {
                DB::rollBack();
            } catch (Throwable $rollbackException) {
                //                throw new QueryException($rollbackException->getMessage(), [], $exception);
            }

            throw $exception;
        }

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'purchase invoice']));
    }

    /**
     * Updates a purchase invoice.
     */
    public function update(StorePurchaseInvoice $request, $purchaseInvoiceId): Response
    {
        $inputs = $request->validated();

        DB::beginTransaction();
        try {
            // Save Purchase Invoice
            $purchaseInvoice = PurchaseInvoice::with([])->findOrFail(e($purchaseInvoiceId));
            // Prevent the resetting of purchase order id
            unset($inputs['purchase_order_id']);
            $purchaseInvoice->fill($inputs);
            $purchaseInvoice->save();

            if (! empty($inputs['purchase_invoice_lines'])) {
                // Aggregate invoice lines
                $inputs['purchase_invoice_lines'] = PurchaseInvoice::aggregateInvoiceLines($inputs['purchase_invoice_lines']);

                $purchaseInvoiceLines = $inputs['purchase_invoice_lines'];

                // update/create purchase order lines(without delete)
                $purchaseInvoiceLines = $purchaseInvoice->purchaseOrder->setPurchaseOrderLines($purchaseInvoiceLines, false);

                // save purchase invoice lines
                $purchaseInvoice->syncInvoiceLines($purchaseInvoiceLines);

                $purchaseInvoice->purchaseOrder->invoiced($purchaseInvoice->purchase_invoice_date);
            }

            $purchaseInvoice->loadMissing('purchaseInvoiceLines');

            $this->response->addData(PurchaseInvoiceResource::make($purchaseInvoice));

            DB::commit();
        } catch (Throwable $exception) {
            try {
                DB::rollBack();
            } catch (Throwable $rollbackException) {
                //                throw new QueryException($rollbackException->getMessage(), [], $exception);
            }

            throw $exception;
        }

        return $this->response->setMessage(__('messages.success.update', ['resource' => 'purchase invoice']));
    }

    /**
     * Store custom Purchase invoice like fees and fire events to recalculate fifo layers and product average.
     *
     *
     * @throws Throwable
     */
    public function storeCustom(StoreCustomPurchaseInvoice $request): Response
    {
        $inputs = $request->all();

        DB::transaction(function () use ($inputs) {
            /**
             * Save Purchase Invoice.
             */
            $purchaseInvoice = new PurchaseInvoice($inputs);
            $purchaseInvoice->supplier_id = $inputs['supplier_id'];
            $purchaseInvoice->save();

            /**
             * custom purchase invoice line.
             */
            $customInvoiceLine = new PurchaseInvoiceLine([
                'description' => $inputs['description'],
                'nominal_code' => 511, // static ??
                'quantity_invoiced' => 1,
            ]);
            $customInvoiceLine->total = $inputs['total'];

            /**
             * Save purchase invoice line.
             */
            $purchaseInvoice->purchaseInvoiceLines()->save($customInvoiceLine);

            /**
             * Fire event to log activity and recalculate fifo layers total cost and product average cost.
             */
            event(new CustomPurchaseInvoiceCreated($purchaseInvoice));
        });

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'custom Purchase Invoice']));
    }

    public function destroy($invoiceId)
    {
        $invoice = PurchaseInvoice::with([])->findOrFail(e($invoiceId));

        return DB::transaction(function () use ($invoice) {
            $invoice->delete();

            return $this->response->setMessage(__('messages.success.delete', ['resource' => 'Purchase Invoice']));
        });
    }

    /**
     * bulk delete using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:purchase_invoices,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];

        foreach ($ids as $index => $id) {
            $result[$index] = ['id' => $id];
            $result[$index]['deletable'] = true; // Invoices are always deletable
            $result[$index]['reason'] = null;
        }

        return $this->response->addData($result);
    }
}
