<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreProductPricingTier;
use App\Http\Resources\ProductPricingTierResource;
use App\Models\ProductPricingTier;
use App\Response;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Throwable;

class ProductPricingTierController extends Controller
{
    use BulkOperation
  {
      archiveInstance as originalArchiveInstance;
  }
    use DataTable, ImportsData;

    protected $model_path = ProductPricingTier::class;

    private $resource = 'product pricing tier';

    /**
     * Display the product pricing tier.
     */
    public function show(ProductPricingTier $productPricingTier): Response
    {
        return $this->response->addData(ProductPricingTierResource::make($productPricingTier));
    }

    /**
     * @throws Throwable
     */
    public function store(StoreProductPricingTier $request): JsonResponse
    {
        $productPricingTier = new ProductPricingTier($request->all());
        $productPricingTier->currency_code = $request->input('currency_code');
        $productPricingTier->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(ProductPricingTierResource::make($productPricingTier));
    }

    public function update(StoreProductPricingTier $request, ProductPricingTier $productPricingTier): JsonResponse
    {
        $productPricingTier->name = $request->input('name');
        $productPricingTier->currency_code = $request->input('currency_code');
        $productPricingTier->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $productPricingTier->name,
            ]))
            ->addData(ProductPricingTierResource::make($productPricingTier));
    }

    /**
     * Archive the product pricing tier.
     */
    public function archive(ProductPricingTier $productPricingTier): JsonResponse
    {
        $this->response->addData(ProductPricingTierResource::make($productPricingTier));

        // prevent archive the default pricing tier
        if ($productPricingTier->is_default) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.product.archive_pricing_failed', ['id' => $productPricingTier->name]), 'ProductPricingTier'.Response::CODE_UNACCEPTABLE, 'id', ['id' => $productPricingTier->id]);
        }

        if ($productPricingTier->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $productPricingTier->name,
                ]));
        }

        return $this->response
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $productPricingTier->name,
            ]), 'ProductPricingTier'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $productPricingTier->id]);
    }

    /**
     * Unarchived the product pricing tier.
     */
    public function unarchived(ProductPricingTier $productPricingTier): JsonResponse
    {
        if ($productPricingTier->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $productPricingTier->name,
                ]))
                ->addData(ProductPricingTierResource::make($productPricingTier));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $productPricingTier->name,
            ]), 'ProductPricingTier'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $productPricingTier->id])
            ->addData(ProductPricingTierResource::make($productPricingTier));
    }

    /**
     * Set product pricing tier as default tier.
     */
    public function setAsDefault(ProductPricingTier $productPricingTier): JsonResponse
    {
        // check is archived
        if ($productPricingTier->isArchived()) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.product.set_default_pricing_failed', ['id' => $productPricingTier->name]), 'ProductPricingTier'.Response::CODE_UNACCEPTABLE, 'id', ['id' => $productPricingTier->id])
                ->addData(ProductPricingTierResource::make($productPricingTier));
        }

        // set default to true
        $productPricingTier->is_default = true;
        $productPricingTier->save();

        return $this->response
            ->setMessage(__('messages.success.default', [
                'resource' => $this->resource,
                'id' => $productPricingTier->name,
            ]))
            ->addData(ProductPricingTierResource::make($productPricingTier));
    }

    /**
     * Delete Product Pricing Tier.
     *
     *
     * @throws Exception
     */
    public function destroy(ProductPricingTier $productPricingTier): Response
    {
        $reasons = $productPricingTier->delete();

        // check if the resource is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['product_pricing_tier_id' => $productPricingTier->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => $this->resource,
                    'id' => $productPricingTier->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $productPricingTier->name,
        ]));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Bulk delete resources.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:product_pricing_tiers,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $productPricingTiers = ProductPricingTier::with([])->whereIn('id', $ids)->select('id', 'name', 'is_default')->get();
        foreach ($productPricingTiers as $key => $productPricingTier) {
            $isUsed = $productPricingTier->isUsed();

            $result[$key] = $productPricingTier->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * check the possibility of archiving.
     */
    public function isArchivable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:product_pricing_tiers,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $productPricingTiers = ProductPricingTier::with([])->whereIn('id', $ids)->select('id', 'name', 'is_default')->get();
        foreach ($productPricingTiers as $key => $productPricingTier) {
            $result[$key] = $productPricingTier->only('id', 'name', 'is_default');
            $result[$key]['archivable'] = ! $productPricingTier->is_default;
            $result[$key]['reason'] = $productPricingTier->is_default ? ['is_default' => __('messages.product.archive_pricing_failed', ['id' => $productPricingTier->name])] : null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function archiveInstance(Model $instance)
    {
        // prevent archive the default pricing tier
        if ($instance->is_default) {
            $this->response->addWarning(__('messages.product.archive_pricing_failed', ['id' => $instance->name]), 'ProductPricingTier'.Response::CODE_UNACCEPTABLE, "ids.{$instance->id}.id", ['id' => $instance->id]);

            return false;
        }

        return $this->originalArchiveInstance($instance);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return ProductPricingTier::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return ProductPricingTierResource::class;
    }
}
