<?php

namespace App\Http\Controllers;

use App\Abstractions\Integrations\SalesChannels\ProductListingResourceInterface;
use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Http\Requests\UpdateProductListing;
use App\Http\Resources\ProductListingResource;
use App\Models\Product;
use App\Models\ProductListing;
use App\Response;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class ProductListingsController extends Controller
{
    use DataTable;

    /**
     * Display a list of resource.
     *
     *
     * @return JsonResponse|AnonymousResourceCollection
     */
    public function index(Product $product, Request $request)
    {
        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications($this->getModel()));
        }

        // prevent send included and excluded together
        if ($request->has('included') and $request->has('excluded')) {
            return $this->response->error(Response::HTTP_BAD_REQUEST, [__('messages.failed.not_both_include_exclude')])
                ->setMessage(__('messages.failed.not_both_include_exclude'));
        }

        // set default included
        $this->setDefaultIncludedColumns($request);

        /**
         * model with relations.
         */
        $builder = ProductListing::with(['salesChannel.integrationInstance.integration', 'productPricingTier'])
            ->filter($request)
            ->addRelations($request)
            ->addCustomColumns()
            ->sort($request)
            ->archived($request->get('archived', 0));
        // to force using AND conjunction
        $builder->where('product_id', $product->id);

        // paginate with limit per page (default 10)
        $builder = DataTableConfiguration::paginate($builder);
        return ProductListingResource::collectionWithTableSpecifications($builder, $this->getModel());
    }

    /**
     * Show a product listing.
     */
    public function show(ProductListing $productListing): ProductListingResourceInterface
    {
        $productListing->load('salesChannel.integrationInstance', 'productPricingTier','inventoryLocations');

        $modelPath = $productListing->salesChannel->integration->getProductsModelPath();
        return $modelPath::getProductListingResource()::make($productListing);
    }

    /**
     * Update a product listing.
     */
    public function update(UpdateProductListing $request, ProductListing $productListing): Response
    {
        $productListing->fill($request->validated());
        // only update price or quantity if they need to update
        $saveOptions = ['set_price' => false, 'set_quantity' => false];
        if ($productListing->isDirty(['master_of_price', 'product_pricing_tier_id'])) {
            $saveOptions['set_price'] = true;
        }
        if ($request->has('inventoryLocations')) {
            $saveOptions['set_quantity'] = true;
        }

        $productListing->save($saveOptions);
        return $this->response->setMessage(__('messages.success.update', [
            'resource' => 'Product Listing',
            'id' => $productListing->sales_channel_listing_id,
        ]))->addData(ProductListingResource::make($productListing));
    }

    protected function getModel(): string
    {
        return ProductListing::class;
    }

    protected function getResource(): string
    {
        return ProductListingResource::class;
    }
}
