<?php

namespace App\Http\Controllers;

use App\Helpers;
use App\Http\Requests\StoreProductImage;
use App\Http\Requests\StoreProductImages;
use App\Models\Product;
use App\Models\ProductImage;
use App\Response;
use Exception;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ProductImageController extends Controller
{
    /**
     * Display product images.
     */
    public function index(Product $product): Response
    {
        return $this->response->addData($product->images);
    }

    /**
     * @throws FileNotFoundException
     */
    public function bulkUpdateCreate(StoreProductImages $request, Product $product): Response
    {
        $product->addImages($request->input('images'));

        return $this->response->setMessage(__('messages.product.update_images'))
            ->addData($product->images);
    }

    /**
     * Add a new image to product.
     *
     *
     * @throws FileNotFoundException
     */
    public function store(StoreProductImage $request, Product $product): Response
    {
        $imageAttributes = $request->all();

        // store image
        $imageUrl = Helpers::getImageUrl($imageAttributes['image'], boolval($imageAttributes['download'] ?? false));

        if ($imageUrl === false) {
            // add warning
            $primaryImage = boolval($imageAttributes['is_primary'] ?? false);
            $message = $primaryImage ?
        __('messages.product.invalid_main_image_url', ['url' => $imageAttributes['image']]) :
        __('messages.product.invalid_image_url', ['url' => $imageAttributes['image'], 'index' => null]);

            $data = ['url' => $imageAttributes['image'], 'main_image' => $primaryImage];

            $this->response->addWarning($message, Response::CODE_UNRESOLVABLE_IMAGE, 'image', $data);
        }

        $imageAttributes['url'] = $imageUrl ?: $imageAttributes['image'];
        $imageAttributes['resolvable'] = boolval($imageUrl);

        $productImage = $product->images()->create($imageAttributes);

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'product image']))
            ->addData($productImage);
    }

    /**
     * Update product image.
     *
     *
     * @throws FileNotFoundException
     */
    public function update(StoreProductImage $request, ProductImage $productImage): Response
    {
        $imageAttributes = $request->all();

        if (! empty($imageAttributes['image'])) {
            // store image
            $imageUrl = Helpers::getImageUrl($imageAttributes['image'], boolval($imageAttributes['download'] ?? false));

            if ($imageUrl === false) {
                // add warning
                $primaryImage = boolval($imageAttributes['is_primary'] ?? false);
                $message = $primaryImage ?
          __('messages.product.invalid_main_image_url', ['url' => $imageAttributes['image']]) :
          __('messages.product.invalid_image_url', ['url' => $imageAttributes['image'], 'index' => null]);

                $data = ['id' => $productImage->id, 'url' => $imageAttributes['image'], 'main_image' => $primaryImage];

                $this->response->addWarning($message, Response::CODE_UNRESOLVABLE_IMAGE, 'image', $data);
            }

            $imageAttributes['url'] = $imageUrl ?: $imageAttributes['image'];
            $imageAttributes['resolvable'] = boolval($imageUrl);
        }

        $productImage->fill($imageAttributes);
        $productImage->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => 'product image',
                'id' => $productImage->id,
            ]))
            ->addData($productImage);
    }

    /**
     * Delete product image.
     *
     *
     * @throws Exception
     */
    public function destroy(ProductImage $productImage): Response
    {
        $productImage->delete();

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'product image',
            'id' => $productImage->id,
        ]));
    }

    /**
     * Store image to server (base64 or url).
     *
     *
     * @return Response contains <URL> of stored image
     *
     * @throws FileNotFoundException
     */
    public function storeImageToServer(Request $request): Response
    {
        $request->validate([
            'image' => 'required',
        ]);

        $imageUrl = Helpers::getImageUrl($request->hasFile('image') ? $request->file('image') : $request->input('image'), true);

        if ($imageUrl) {
            return $this->response->addData(['url' => Storage::disk('images')->url($imageUrl)])
                ->setMessage(__('messages.success.store_image'));
        }

        return $this->response->error(Response::HTTP_BAD_REQUEST)
            ->addError(__('messages.product.invalid_image_store'), Response::CODE_UNRESOLVABLE_IMAGE, 'image', ['image' => $request->input('image')]);
    }

    /**
     * Remove image from server by path.
     */
    public function deleteImageFromServer(Request $request): Response
    {
        $request->validate([
            'url' => 'required',
        ]);

        // delete the image and its thumbnail
        if (Helpers::deleteImage($request->input('url'))) {
            return $this->response->setMessage(__('messages.success.delete_image'));
        }

        return $this->response->error(Response::HTTP_BAD_REQUEST)
            ->addError(__('messages.failed.delete_image'), Response::CODE_NOT_FOUND, 'url', ['url' => $request->input('url')])
            ->setMessage(__('messages.failed.delete_image'));
    }
}
