<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Resources\ProductBrandResource;
use App\Models\ProductBrand;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProductBrandController extends Controller
{
    use BulkOperation, DataTable, ImportsData;

    protected $model_path = ProductBrand::class;

    private $resource = 'product brand';

    /**
     * Get product brand.
     */
    public function show(ProductBrand $productBrand): Response
    {
        return $this->response->addData(ProductBrandResource::make($productBrand));
    }

    /**
     * Create New product brand.
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate(
            [
                'name' => 'required|max:255|unique:product_brands,name',
            ]
        );

        $productBrand = new ProductBrand($request->all());
        $productBrand->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(ProductBrandResource::make($productBrand));
    }

    /**
     * Update a product brand.
     */
    public function update(Request $request, $id): JsonResponse
    {
        $request->validate(
            [
                'name' => 'required|max:255|unique:product_brands,name,'.$id,
            ]
        );

        $productBrand = ProductBrand::with([])->findOrFail($id);
        $productBrand->fill($request->all());
        $productBrand->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $productBrand->name,
            ]))
            ->addData(ProductBrandResource::make($productBrand));
    }

    /**
     * Archive the product brand.
     */
    public function archive($productBrandId): JsonResponse
    {
        $productBrand = ProductBrand::with([])->findOrFail($productBrandId);

        if ($productBrand->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $productBrand->name,
                ]))
                ->addData(ProductBrandResource::make($productBrand));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $productBrand->name,
            ]), 'ProductBrand'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $productBrand->id])
            ->addData(ProductBrandResource::make($productBrand));
    }

    /**
     * Unarchived the product brand.
     */
    public function unarchived($productBrandId): JsonResponse
    {
        $productBrand = ProductBrand::with([])->findOrFail($productBrandId);

        if ($productBrand->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $productBrand->name,
                ]))
                ->addData(ProductBrandResource::make($productBrand));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $productBrand->name,
            ]), 'ProductBrand'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $productBrand->id])
            ->addData(ProductBrandResource::make($productBrand));
    }

    /**
     * Delete product brand.
     *
     *
     * @throws Exception
     */
    public function destroy(ProductBrand $productBrand): Response
    {
        $reasons = $productBrand->delete();

        // check if the ProductBrand is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['product_brand_id' => $productBrand->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'product brand',
                    'id' => $productBrand->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'product brand',
            'id' => $productBrand->name,
        ]));
    }

    /**
     * Bulk delete nominal codes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * Bulk archive nominal codes.
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * Bulk unarchived nominal codes.
     *
     *
     * @return mixed
     */
    public function bulkUnArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:product_brands,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $productBrands = ProductBrand::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($productBrands as $key => $productBrand) {
            $isUsed = $productBrand->isUsed();

            $result[$key] = $productBrand->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return ProductBrand::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return ProductBrandResource::class;
    }
}
