<?php

namespace App\Http\Controllers;

use App\DataTable\DataTableConfiguration;
use App\Models\AccountingTransaction;
use App\Models\Address;
use App\Models\Attribute;
use App\Models\BackorderQueue;
use App\Models\Concerns\HasFilters;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\CustomField;
use App\Models\FifoLayer;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\NominalCode;
use App\Models\Payment;
use App\Models\ProductAttribute;
use App\Models\ProductBrand;
use App\Models\PurchaseOrder;
use App\Models\SalesChannelType;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLineFinancial;
use App\Models\ShipBySchedule;
use App\Models\ShippingMethod;
use App\Models\StockTake;
use App\Models\Store;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\Tag;
use App\Models\Warehouse;
use App\Queries\Product;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasOneOrMany;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Xero\Entities\XeroTransaction;

class LookupController extends Controller
{
    /**
     * Lookup product inputs.
     *
     * @return array
     */
    public function products()
    {
        $product = new Product();

        $excludedInputs = array_merge($this->getNumricables($product), [
            'inventory_',
        ]);
        $customInputs = [
            'defaultSupplierProduct.supplier.' => Supplier::class,
            'supplierProducts.supplier.' => Supplier::class,
            'productAttributes.' => ProductAttribute::class,
            'productPricingTiers.' => Currency::class,
            'defaultSupplierProduct.supplierPricingTiers' => Currency::class,
            'brand' => ProductBrand::class,
            'salesNominalCode.name' => NominalCode::class,
            'cogsNominalCode.name' => NominalCode::class,
            'tags.name' => Tag::class,

        ];
        $staticInputs = [
            'weight_unit' => Product::WEIGHT_UNITS,
            'dimension_unit' => Product::DIMENSION_UNITS,
            'case_dimensions_unit' => Product::DIMENSION_UNITS,
            'type' => Product::TYPES,
        ];

        return $this->lookup(
            $product,
            $staticInputs,
            $excludedInputs,
            $customInputs,
            function ($builder, $key, $lastKey) {
                // for product attributes
                if ($key === 'productAttributes.') {
                    $attributeId = Attribute::with([])->where('name', $lastKey)->first()->id;
                    $builder->where('attribute_id', $attributeId);

                    return 'value';
                } elseif ($key === 'productPricingTiers.' || $key === 'defaultSupplierProduct.supplierPricingTiers') {
                    if ($lastKey == 'currency_code') {
                        return 'code';
                    }

                    return false;
                }

                return $lastKey;
            }
        );
    }

    /**
     * Lookup sales order inputs.
     *
     * @return array
     */
    public function salesOrders()
    {
        $salesOrder = new SalesOrder();

        $excludedInputs = array_merge($this->getNumricables($salesOrder), ['item_quantity', 'item_price', 'covered_by_po']);
        $customInputs = [
            'salesChannel.type.' => SalesChannelType::class,
            'customer.address.' => Address::class,
            'salesOrderLines.product.brand' => ProductBrand::class,
            'salesOrderLines.product.' => Product::class,
            'salesOrderLines.bundle.' => Product::class,
            'tags.name' => Tag::class,
            'salesOrderLines.nominalCode.' => NominalCode::class,
            'salesOrderLines.warehouse.' => Warehouse::class,
            'salesChannel.integrationInstance' => IntegrationInstance::class,
            'salesChannel.integrationInstance.integration' => Integration::class,
            'shippingMethod.full_name' => ShippingMethod::class,
        ];
        $staticInputs = [
            'order_status' => SalesOrder::STATUSES,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUES,
            'payment_status' => SalesOrder::PAYMENT_STATUSES,
            'fulfillable' => SalesOrder::FULFILLABLE_STATUSES,
        ];

        return $this->lookup(
            $salesOrder,
            $staticInputs,
            $excludedInputs,
            $customInputs,
            null,
            [],
            function ($combinedKey, $model) {
                return $combinedKey === 'salesChannel.integrationInstance.integration.name' ? Integration::with([]) : $model::with([]);
            }
        );
    }

    public function backorderQueues()
    {
        $backorderQueue = new BackorderQueue;

        $excludedInputs = array_merge($this->getNumricables($backorderQueue), [
            'backordered_quantity',
            'released_quantity',
        ]);
        $customInputs = [
            'salesOrderLine.product.' => Product::class,
            'salesOrderLine.salesOrder.' => SalesOrder::class,
            'backorderQueueCoverages.purchaseOrderLine.purchaseOrder.' => PurchaseOrder::class,
        ];
        $staticInputs = [];

        return $this->lookup(
            $backorderQueue,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    public function salesOrderLineFinancials()
    {
        $salesOrderLineFinancial = new SalesOrderLineFinancial();

        $excludedInputs = [];
        $customInputs = [];
        $staticInputs = [];

        return $this->lookup(
            $salesOrderLineFinancial,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    /**
     * Lookup purchase order inputs.
     *
     * @return array
     */
    public function purchaseOrders()
    {
        $purchaseOrder = new PurchaseOrder();

        $excludedInputs = array_merge($this->getNumricables($purchaseOrder), [
            'shipment_item_quantity',
            'shipment_received',
            'item_quantity',
            'item_price',
            'item_received',
            'additional_cost',
            'tax_cost',
            'discount',
            'product_total',
            'total',
        ]);
        $customInputs = [
            'supplier.address' => Address::class,
            'customer.address.' => Address::class,
            'tags.name' => Tag::class,
            'purchaseOrderLines.product.' => Product::class,
            'purchaseOrderLines.nominalCode.' => NominalCode::class,
            'purchaseOrderLines.supplierProducts.' => SupplierProduct::class,
            'purchaseOrderShipments.fulfilledShippingMethod.' => ShippingMethod::class,
        ];
        $staticInputs = [
            'order_status' => PurchaseOrder::STATUS,
            'submission_status' => PurchaseOrder::SUBMISSION_STATUS,
            'receipt_status' => PurchaseOrder::RECEIPT_STATUS,
            'shipment_status' => PurchaseOrder::SHIPMENT_STATUS,
            'invoice_status' => PurchaseOrder::INVOICE_STATUS,
            'submission_format' => PurchaseOrder::SUBMISSION_FORMATS,
        ];

        return $this->lookup(
            $purchaseOrder,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    /**
     * Lookup accounting transaction inputs.
     *
     * @return array
     */
    public function accountingTransactions()
    {
        $accountingTransaction = new AccountingTransaction();

        $excludedInputs = [];
        $customInputs = [
            'accountingTransactionLines.nominalCode.' => NominalCode::class,
        ];
        $staticInputs = [
            'status' => AccountingTransaction::STATUSES,
        ];

        return $this->lookup(
            $accountingTransaction,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    /**
     * Lookup accounting payment inputs.
     *
     * @return array
     */
    public function accountingPayments()
    {
        $payment = new Payment();

        $excludedInputs = [];
        $customInputs = [
            'accountingIntegration' => XeroTransaction::class,
        ];
        $staticInputs = [
            'status' => Payment::STATUSES,
            'type' => AccountingTransaction::TYPES_WITH_PAYMENT,
        ];

        return $this->lookup(
            $payment,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    /**
     * @return array|mixed
     */
    public function stockTakes()
    {
        $stockTake = new StockTake;

        $excludedInputs = array_merge($this->getNumricables($stockTake), [
            'item_quantity',
        ]);
        $customInputs = [
            'stockTakeItems.product' => Product::class,
            'warehouse' => Warehouse::class,
        ];
        $staticInputs = [
            'status' => StockTake::STOCK_TAKE_STATUSES,
        ];

        return $this->lookup(
            $stockTake,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    /**
     * Lookup sales order inputs.
     *
     * @return array
     */
    public function inventoryMovements()
    {
        $inventoryMovement = new InventoryMovement();

        $excludedInputs = $this->getNumricables($inventoryMovement);
        $customInputs = [];
        $staticInputs = [
            'inventory_status' => InventoryMovement::INVENTORY_STATUS,
            'type' => InventoryMovement::TYPES,
        ];

        $containsFilterInputs = ['layer_type', 'link_type'];

        return $this->lookup(
            $inventoryMovement,
            $staticInputs,
            $excludedInputs,
            $customInputs,
            null,
            $containsFilterInputs
        );
    }

    /**
     * Lookup sales order inputs.
     *
     * @return array
     */
    public function fifoLayers()
    {
        $inventoryMovement = new FifoLayer();

        $excludedInputs = $this->getNumricables($inventoryMovement);
        $customInputs = [
            'warehouse' => Warehouse::class,
            'product' => Product::class,
        ];
        $staticInputs = [
            'link_types' => FifoLayer::REQUEST_LINK_TYPES,
        ];

        $containsFilterInputs = ['link_type'];

        return $this->lookup(
            $inventoryMovement,
            $staticInputs,
            $excludedInputs,
            $customInputs,
            null,
            $containsFilterInputs
        );
    }

    public function inventoryAdjustments()
    {
        $inventoryAdjustments = new InventoryAdjustment();

        $excludedInputs = $this->getNumricables($inventoryAdjustments);
        $customInputs = [
            'warehouse' => Warehouse::class,
            'product' => Product::class,
        ];
        $staticInputs = [];

        return $this->lookup(
            $inventoryAdjustments,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    public function integrationListings(IntegrationInstance $integrationInstance)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getProductsModelPath();
        $model = $modelPath::with('productListing')
            ->where('integration_instance_id', $integrationInstance->id)
            ->firstOrFail();

        $excludedInputs = $this->getNumricables($model);
        $customInputs = [
            'product.' => Product::class,
        ];
        $staticInputs = [];

        return $this->lookup(
            $model,
            $staticInputs,
            $excludedInputs,
            $customInputs,
            null
        );
    }

    /**
     * Lookup sales order inputs.
     *
     * @return array
     */
    public function salesOrderFulfillments()
    {
        $salesOrder = new SalesOrderFulfillment();

        $excludedInputs = array_merge($this->getNumricables($salesOrder), ['item_quantity', 'item_price']);
        $customInputs = [
            'salesOrder.store.' => Store::class,
            'salesOrder.customer.' => Customer::class,
            'salesOrder.customer.address.' => Address::class,
            'salesOrderFulfillmentLines.salesOrderLine.product.' => Product::class,
            'salesOrderFulfillmentLines.salesOrderLine.nominalCode.' => NominalCode::class,
            'salesOrder.salesChannel.integrationInstance.integration' => Integration::class,
            'requestedShippingMethod.full_name' => ShippingMethod::class,
            'fulfilledShippingMethod.full_name' => ShippingMethod::class,
            'salesOrder.fulfillment_number' => SalesOrder::class,
            'salesOrder.tags.name' => Tag::class,
        ];
        $staticInputs = [
            'status' => SalesOrderFulfillment::STATUS,
        ];

        return $this->lookup(
            $salesOrder,
            $staticInputs,
            $excludedInputs,
            $customInputs,
            function (Builder $builder, $key, $lastKey) {
                if ($lastKey == 'fulfillment_number') {
                    $builder->join('sales_order_fulfillments', 'sales_order_fulfillments.sales_order_id', '=', 'sales_orders.id');

                    return DB::raw('CONCAT_WS(".", `sales_order_number`, `sales_order_fulfillments`.`fulfillment_sequence`)');
                }

                return $lastKey;
            },
            [],
            function ($combinedKey, $model) {
                return $combinedKey === 'salesChannel.integrationInstance.integration.name' ? Integration::with([]) : $model::with([]);
            }
        );
    }

    /**
     * Lookup tags
     *
     * @return array
     */
    public function tags()
    {
        $tag = new Tag();

        $excludedInputs = [];
        $customInputs = [];
        $staticInputs = [];

        return $this->lookup(
            $tag,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    public function customFields(Request $request): array
    {
        $inputs = $request->validate([
            'link_type' => 'required|string',
        ]);

        return CustomField::query()
            ->select('id', 'name')
            ->where('link_type', $inputs['link_type'])
            ->get()
            ->toArray();
    }

    public function shipBySchedule()
    {
        $shipBySchedule = new ShipBySchedule();

        $excludedInputs = [];
        $customInputs = [
            'warehouse.' => Warehouse::class,
        ];
        $staticInputs = [];

        return $this->lookup(
            $shipBySchedule,
            $staticInputs,
            $excludedInputs,
            $customInputs
        );
    }

    /**
     * Lookup function used to get the suggestion values based on the input and value that send in the request
     *
     * @param  Model  $resource the base model
     * @param  array  $staticInputs fields that get their values from static enums
     * @param  array  $excludedInputs fields that should exclude from lookups (numeric fields)
     * @param  array  $customInputs fields that need to lookup from another model (relational fields)
     * @param  callable|null  $keyForCustomInputs a callback function that used to make a custom query on the relational fields and return the last real column
     * @param  array  $containsFilterInputs fields that should filter by "contains" operator, the default operator is "startsWith"
     * @param  callable|null  $builderForCustomInputs a callback function to customize the model for custom fields
     * @return array|mixed
     */
    private function lookup(
        Model $resource,
        array $staticInputs,
        array $excludedInputs,
        array $customInputs,
        ?callable $keyForCustomInputs = null,
        array $containsFilterInputs = [],
        ?callable $builderForCustomInputs = null
    ) {
        $input = request()->get('input'); // the lookup key, get it from the GET request params
        $value = request()->get('value'); // the lookup value, get it from the GET request params
        $limit = request()->get('limit', 10);

        if (empty($input)) {
            return ['data' => []];
        }

        // handle static fields to get from enums
        if (in_array($input, array_keys($staticInputs))) {
            return ['data' => $staticInputs[$input]];
        }

        // handle excluded fields
        if (Str::startsWith($input, $excludedInputs)) {
            return ['data' => []];
        }

        $realKey = DataTableConfiguration::getRealKey(get_class($resource), $input);

        // handle relational fields
        if ($relation = $this->isRelation($realKey, $resource)) {
            // use the customization if the field is in customInputs
            foreach ($customInputs as $key => $model) {
                if (Str::startsWith($relation['combined_key'], $key)) {
                    $lastKey = Arr::last(explode('.', $relation['combined_key']));

                    // check model customization
                    /** @var Builder $builder */
                    $builder = $builderForCustomInputs ? $builderForCustomInputs($relation['combined_key'], $model) : $model::with([]);

                    if ($keyForCustomInputs) {
                        $lastKey = $keyForCustomInputs($builder, $key, $lastKey);
                    }

                    if ($lastKey === false) {
                        return ['data' => []];
                    }

                    if (is_string($lastKey)) {
                        $lastKey = DataTableConfiguration::getRealKey($model, $lastKey);
                    }

                    // get the data from the customized query builder then flatten it
                    return $this->flattenPaginationItems($builder->select($lastKey)
                        ->where($lastKey, 'like', in_array($realKey, $containsFilterInputs) ? "%$value%" : "$value%")
                        ->whereNotNull($lastKey)
                        ->where($lastKey, '!=', '')
                        ->distinct($lastKey)
                        ->paginate($limit));
                }
            }

            // get the data based on the query builder that get from the model relation
            return $this->flattenPaginationItems($this->filterRelation($resource, $relation, $value));
        }

        // non-relational fields, build the query from the base model

        $realKey = is_array($realKey) ? $realKey['key'] : $realKey;

        // get the data from the base model then flatten it
        return $this->flattenPaginationItems($resource::with([])->select($realKey)
            ->where($realKey, 'like', in_array($realKey, $containsFilterInputs) ? "%$value%" : "$value%")
            ->whereNotNull($realKey)
            ->where($realKey, '!=', '')
            ->distinct($realKey)
            ->paginate($limit));
    }

    /**
     * Lookup standard relation.
     */
    private function filterRelation(Model $model, $relation, $value): LengthAwarePaginator
    {
        // get the additional filters(wheres) that be in the relation definition
        if ($model->{$relation['name']}() instanceof BelongsToMany) {
            //        $excludeKey = 'getQualifiedForeignPivotKeyName';
            $wheres = [];
        } else {
            // exclude the foregin key from the filters
            $excludeKey = $model->{$relation['name']}() instanceof HasOneOrMany ? 'getQualifiedForeignKeyName' : 'getQualifiedOwnerKeyName';

            $wheres = collect($model->{$relation['name']}()->getQuery()->getQuery()->wheres);
            $wheres = $wheres->where('column', '!=', $model->{$relation['name']}()->{$excludeKey}());
            $wheres = $wheres->values()->toArray();
        }

        // get the query builder from model that related of the relation
        /** @var Builder $builder */
        $builder = $model->{$relation['name']}()->getRelated()->select($relation['key'])
            ->where($relation['key'], 'like', $value.'%')
            ->whereNotNull($relation['key'])
            ->where($relation['key'], '!=', '')
            ->distinct($relation['key']);

        // add the additional filters(wheres)
        foreach ($wheres as $where) {
            if ($where['type'] === 'Null') {
                $builder->whereNull($where['column']);
            } elseif ($where['type'] === 'Basic') {
                $builder->where($where['column'], $where['value']);
            }
        }

        return $builder->paginate(request('limit', 10));
    }

    /**
     * Determined whether key is relation.
     *
     * @param  string|array  $key the real key
     * @param  Model|HasFilters  $model instance of model that use HasFilters trait
     * @return bool|array Array means the key is from a relation, Boolean: false only
     */
    private function isRelation(
        string|array $key,
        Model $model
    ): bool|array {
        if (is_array($key)) {
            $relation = $key['is_relation'] ? $model->isRelationByDot($key['key']) : false;
        } else {
            $relation = $model->isRelationByDot($key);
        }

        return $relation;
    }

    /**
     * Get columns that castable to numeric value.
     *
     * @param  Model  $model instance of model
     */
    private function getNumricables(Model $model): array
    {
        return collect($model->getCasts())->whereIn(null, [
            'int',
            'integer',
            'real',
            'float',
            'double',
            'decimal',
            'bool',
            'boolean',
        ])->keys()->toArray();
    }

    /**
     * Flatten pagination items to return without keys.
     *
     *
     * @return mixed
     */
    private function flattenPaginationItems(LengthAwarePaginator $paginator)
    {
        $paginationData = $paginator->toArray();

        $paginationData['data'] = Arr::flatten($paginationData['data']);

        return $paginationData;
    }

}
