<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\InsufficientStockException;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Http\Resources\InventoryAdjustmentResource;
use App\Models\InventoryAdjustment;
use App\Repositories\InventoryAdjustmentRepository;
use App\Response;
use App\Services\InventoryAdjustment\InventoryAdjustmentService;
use App\Services\InventoryAdjustment\UpdateInventoryAdjustmentService;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Throwable;

/**
 * Class InventoryAdjustmentController.
 */
class InventoryAdjustmentController extends Controller
{
    use BulkOperation, DataTable;

    /**
     * @var InventoryAdjustmentRepository
     */
    private $adjustments;

    protected $model_path = InventoryAdjustment::class;

    protected $resource = 'inventory adjustment';

    /**
     * InventoryAdjustmentController constructor.
     */
    public function __construct(InventoryAdjustmentRepository $adjustments)
    {
        parent::__construct();
        $this->adjustments = $adjustments;
    }

    public function show(InventoryAdjustment $inventoryAdjustment): Response
    {
        $inventoryAdjustment->load('fifoLayers');
        $inventoryAdjustment->load(DataTableConfiguration::getRequiredRelations(InventoryAdjustment::class));

        return $this->response->addData(InventoryAdjustmentResource::make($inventoryAdjustment));
    }

    /**
     * Adjust Inventory.
     *
     *
     * @throws Throwable
     */
    public function store(StoreInventoryAdjustment $request): JsonResponse
    {
        try {
            $adjustment = InventoryAdjustmentService::factory($request)->execute();

            if (! $adjustment) { // the adjusted quantity equals to actual inventory count of the product
                return $this->response->addWarning(__('messages.inventory.adjustment_quantity_equals_actual'), Response::CODE_SAME_QUANTITY, 'quantity');
            }

            $adjustment->load(DataTableConfiguration::getRequiredRelations(InventoryAdjustment::class));
            $adjustment->load('fifoLayers');

            return $this->response
                ->setMessage(__('messages.inventory.inventory_adjusted'))
                ->addData(InventoryAdjustmentResource::make($adjustment));
        } catch (InsufficientStockException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(
                    __('messages.inventory.fifo_layer_not_existed', ['product_sku' => $e->productId]),
                    Response::CODE_NO_ENOUGH_QUANTITY,
                    'product_id'
                );
        }
    }

    public function update(StoreInventoryAdjustment $request, InventoryAdjustment $inventoryAdjustment): JsonResponse
    {
        try {
            $adjustment = UpdateInventoryAdjustmentService::make($request)->execute($inventoryAdjustment);

            $adjustment->load(DataTableConfiguration::getRequiredRelations(InventoryAdjustment::class));
            $adjustment->load('fifoLayers');

            return $this->response
                ->setMessage(__('messages.inventory.inventory_adjusted'))
                ->addData(InventoryAdjustmentResource::make($adjustment));
        } catch (InsufficientStockException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(
                    __('messages.inventory.fifo_layer_not_existed', ['product_sku' => $e->productId]),
                    Response::CODE_NO_ENOUGH_QUANTITY,
                    'product_id'
                );
        }
    }

    /**
     * Delete inventory adjustment.
     *
     *
     * @throws Throwable
     */
    public function destroy(InventoryAdjustment $inventoryAdjustment): Response
    {
        $reasons = null;
        try {
            $reasons = $inventoryAdjustment->delete();
        } catch (InsufficientStockException $exception) {
            $reasons = [];
        }

        // check if the inventory adjustment has fifo-layer uncovered fulfilled quantity
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['inventory_adjustment_id' => $inventoryAdjustment->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => $this->resource,
                    'id' => $inventoryAdjustment->id,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $inventoryAdjustment->id,
        ]));
    }

    /**
     * Bulk delete inventory adjustment.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:inventory_adjustments,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        // All warehouse transfers are deletable,
        // only reverses inventory movements.

        $result = [];
        $adjustments = InventoryAdjustment::with([])->whereIn('id', $ids)->select('id', 'quantity', 'product_id')->get();
        /**
         * @var InventoryAdjustment  $adjustment
         */
        foreach ($adjustments as $key => $adjustment) {
            $isDeletable = $adjustment->isDeletable();

            $result[$key] = $adjustment->only('id');
            $result[$key]['deletable'] = $isDeletable;
            $result[$key]['reason'] = $isDeletable ? null : ['fulfilledQuantity' => __('messages.inventory.fulfilled_quantity_not_covered')];
        }

        return $this->response->addData($result);
    }

    public function archive(InventoryAdjustment $adjustment): Response
    {
        if ($adjustment->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $adjustment->id,
                ]))
                ->addData(InventoryAdjustmentResource::make($adjustment));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $adjustment->id,
            ]), 'WarehouseTransfer'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $adjustment->id])
            ->addData(InventoryAdjustmentResource::make($adjustment));
    }

    public function unarchived(InventoryAdjustment $adjustment): Response
    {
        if ($adjustment->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $adjustment->id,
                ]))
                ->addData(InventoryAdjustmentResource::make($adjustment));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $adjustment->id,
            ]), 'WarehouseTransfer'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $adjustment->id])
            ->addData(InventoryAdjustmentResource::make($adjustment));
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return $this->model_path;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return InventoryAdjustmentResource::class;
    }
}
