<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Exceptions\IntegrationInstance\Magento\BundleComponentUnavailableException;
use App\Exceptions\IntegrationInstance\Magento\UnableToCreateBundleProductException;
use App\Http\Requests\ListingsToSkuProductsRequest;
use App\Integrations\Listing;
use App\Jobs\CreateProductsFromListing;
use App\Jobs\ImportListingMapping;
use App\Models\DataImportMapping;
use App\Models\IntegrationInstance;
use App\Models\TaskStatus\TaskStatus;
use App\Response;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class IntegrationInstanceListingController extends IntegrationInstanceController
{
    use DataTable;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    public function exportListings(Request $request, IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;

        return $this->export($request, [
            [
                'field' => 'integration_instance_id',
                'operator' => '=',
                'value' => $integrationInstance->id,
            ],
        ]);
    }

    public function importListingsMappings(IntegrationInstance $integrationInstance, Request $request): JsonResponse
    {
        // Prepare TaskStatus record to report background progress to the UI:
        $task = new TaskStatus();
        $task->title = 'Importing mappings';
        $task->addMessage('File uploaded successfully:  '.$request->input('original_name'));
        dispatch(
            new ImportListingMapping($integrationInstance, $this->getListingFilePath($request), $task)
        )->onQueue('sales-channels');

        return response()->json(['task_id' => $task->id]);
    }

    private function getListingFilePath(Request $request): string
    {
        $originalFilename = $request->input('original_name');
        $filename = $request->input('stored_name');
        // Prepare TaskStatus record to report background progress to the UI:
        $task = new TaskStatus();
        $task->title = 'Importing Amazon mappings';

        $task->addMessage('File uploaded successfully:  '.$originalFilename);

        // Find out real file path
        $uploader_info = config('uploader.listing-mappings', ['target' => 'imports/listing_mappings']);

        return storage_path(rtrim($uploader_info['target'], '/').'/'.$filename);
    }

    public function getMappingInfo(Request $request, IntegrationInstance $integrationInstance)
    {
        /** @var Model $productsPath */
        $productsPath = $integrationInstance->integration->getProductsModelPath();

        $query = $productsPath::with(['productListing'])
            ->where('integration_instance_id', $integrationInstance->id)
            ->where(function ($builder) use ($request) {
                $builder->filter($request);
            })
            ->addRelations($request)
            ->addCustomColumns($request);

        $total = $query->count();
        $mapped = $query->whereHas('productListing')->count();

        return $this->response->addData([
            'mapped' => $mapped,
            'total' => $total,
        ]);
    }

    protected function persistMappings(array $mappings = [], ?IntegrationInstance $instance = null)
    {
        if (! empty($mappings) && $instance) {
            DataImportMapping::setForModel('listings', $mappings, [
                'integration_instance_id' => $instance->id,
            ]);
        }
    }

    public function listingsToSKUProducts(ListingsToSkuProductsRequest $request, IntegrationInstance $integrationInstance)
    {
        $payload = $request->validated();

        if (! empty($payload['filters'])) {
            // We're creating by filters, we get the data by filters

            /** @var Model $productsPath */
            $productsPath = $integrationInstance->integration->getProductsModelPath();

            $query = $productsPath::with(['productListing'])
                ->where('integration_instance_id', $integrationInstance->id)
                ->filter($request)
                ->addRelations($request)
                ->addCustomColumns($request);

            if ($query->count() > 10) {
                $job = CreateProductsFromListing::withFilters(json_decode($payload['filters'], true), $integrationInstance, $payload['mapping']);
                dispatch($job)->onQueue('sales-channels');
                // Save the field mapping
                $this->persistMappings($payload['mapping'] ?? [], $integrationInstance);

                return $this->response->setMessage('Added to queue, products will be created shortly.');
            }

            return $this->createFromQuery($query, $payload['mapping'] ?? [], $integrationInstance);
        } elseif (! empty($payload['ids'])) {
            /** @var Model $productsPath */
            $productsPath = $integrationInstance->integration->getProductsModelPath();

            $ids = array_unique($request->input('ids', []));
            $query = $productsPath::with(['productListing'])
                ->where('integration_instance_id', $integrationInstance->id)
                ->whereIn('_id', $ids);

            if ($query->count() > 10) {
                $job = CreateProductsFromListing::withIds($payload['ids'], $integrationInstance, $payload['mapping']);
                dispatch($job)->onQueue('sales-channels');
                // Save the field mapping
                $this->persistMappings($payload['mapping'] ?? [], $integrationInstance);

                return $this->response->setMessage('Added to queue, products will be created shortly.');
            }

            return $this->createFromQuery($query, $payload['mapping'] ?? [], $integrationInstance);
        } elseif (isset($payload['initial_mapping']) && $payload['initial_mapping']) {
            // Save the field mapping
            $this->persistMappings($payload['mapping'] ?? [], $integrationInstance);

            return $this->response->setMessage('Field mappings saved .');
        } else {
            return $this->response->addError(
                'Need to create products from ids or by filters.',
                Response::CODE_UNACCEPTABLE,
                'IntegrationListings'
            );
        }
    }

    private function createFromQuery($query, array $mapping = [], ?IntegrationInstance $integrationInstance = null)
    {
        $totalAffected = $query->count();

        $cursor = $query->cursor();

        $failedCount = 0;
        /** @var Listing $listing */
        foreach ($cursor as $listing) {
            try {
                if (is_array($result = $listing->createSkuProduct($mapping)) && isset($result['reason'])) {
                    $this->response->addWarning($result['reason'], Response::CODE_UNACCEPTABLE, $result['id'], ['id' => $result['id']]);
                    $failedCount++;
                }
            } catch (UnableToCreateBundleProductException|BundleComponentUnavailableException $exception) {
                $this->response->addWarning($exception->getMessage(), $exception->getResponseErrorCode(), $listing['id']);
                $failedCount++;
            }
        }

        // Save the field mapping
        $this->persistMappings($mapping, $integrationInstance);

        return $this->response->setMessage(__('messages.success.bulk_create', [
            'total_count' => $totalAffected,
            'success_count' => ($totalAffected - $failedCount),
            'resource' => 'product',
        ]));
    }

    protected function getModel()
    {
        return $this->integrationInstance->integration->getProductsModelPath();
    }

    protected function getResource()
    {
        return $this->getModel()::getResource();
    }

    public function getExportFilename()
    {
        return 'Listings for '.$this->integrationInstance->name;
    }
}
