<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Resources\FifoLayerResource;
use App\Http\Resources\InventoryMovementResource;
use App\Models\FifoLayer;
use App\Repositories\FifoLayerRepository;
use App\Response;
use Exception;
use Illuminate\Http\Request;

class FifoLayerController extends Controller
{
    use BulkOperation, DataTable
  {
      archiveInstance as originalArchiveInstance;
  }

    /**
     * @var string
     */
    protected $model_path = FifoLayer::class;

    /**
     * @var string
     */
    protected $resource = 'fifo layer';

    public function __construct(private readonly FifoLayerRepository $fifoLayerRepository)
    {
        parent::__construct();
    }

    public function show(FifoLayer $fifoLayer): Response
    {
        $fifoLayer->load(DataTableConfiguration::getRequiredRelations($this->model_path));

        return $this->response->addData(FifoLayerResource::make($fifoLayer));
    }

    public function fetchForProduct(Request $request, $productId): Response
    {
        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications(FifoLayer::class));
        }

        // prevent send included and excluded together
        if ($request->has('included') and $request->has('excluded')) {
            return $this->response->error(Response::HTTP_BAD_REQUEST, [__('messages.failed.not_both_include_exclude')])
                ->setMessage(__('messages.failed.not_both_include_exclude'));
        }

        // set default included
        $this->setDefaultIncludedColumns($request, FifoLayer::class);

        $fifoLayers = FifoLayer::with([])->where('product_id', e($productId));

        $fifoLayers->filter($request);
        $fifoLayers->addRelations($request);
        $fifoLayers->addCustomColumns();
        $fifoLayers->sort($request);

        // paginate with limit per page (default 10)
        $fifoLayers = DataTableConfiguration::paginate($fifoLayers);

        return $this->response->addData(FifoLayerResource::collectionWithTableSpecifications($fifoLayers, FifoLayer::class));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    public function archive(FifoLayer $fifoLayer): Response
    {
        // Only fully utilized fifo layers can be archived.
        if ($fifoLayer->available_quantity !== 0) {
            return $this->response->addError(__('messages.fifo_layer.archive_not_fully_fulfilled', [
                'resource' => $this->resource,
                'id' => $fifoLayer->id,
            ]), 'FifoLayer'.Response::CODE_NOT_FULLY_FULFILLED, 'id', ['id' => $fifoLayer->id]);
        }

        if ($fifoLayer->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $fifoLayer->id,
                ]))
                ->addData(FifoLayerResource::make($fifoLayer));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $fifoLayer->id,
            ]), 'FifoLayer'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $fifoLayer->id])
            ->addData(FifoLayerResource::make($fifoLayer));
    }

    protected function archiveInstance(FifoLayer $instance): bool
    {
        // Only fully utilized fifo layers can be archived.
        if ($instance->available_quantity > 0) {
            $this->response->error()
                ->addError(__('messages.fifo_layer.archive_not_fully_fulfilled', [
                    'resource' => $this->resource,
                    'id' => $instance->id,
                ]), 'FifoLayer'.Response::CODE_NOT_FULLY_FULFILLED, 'id', ['id' => $instance->id]);

            return false;
        }

        if (! $instance->archive()) {
            // FIFO layer is likely already archived.
            $this->response->warning()
                ->addWarning(__('messages.failed.already_archive', [
                    'resource' => $this->resource,
                    'id' => $instance->id,
                ]), 'FifoLayer'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $instance->id])
                ->addData(FifoLayerResource::make($instance));

            return false;
        }

        // All good!
        return true;
    }

    public function unarchived(FifoLayer $fifoLayer): Response
    {
        if ($fifoLayer->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $fifoLayer->id,
                ]))
                ->addData(FifoLayerResource::make($fifoLayer));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $fifoLayer->id,
            ]), 'FifoLayer'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $fifoLayer->id])
            ->addData(FifoLayerResource::make($fifoLayer));
    }

    /**
     * check the possibility of archiving.
     */
    public function isArchivable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:fifo_layers,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $fifoLayers = FifoLayer::with([])->whereIn('id', $ids)->get();
        foreach ($fifoLayers as $key => $fifoLayer) {
            $archivable = true;

            // is not fully used
            if ($fifoLayer->available_quantity > 0) {
                $archivable = false;
            }

            $result[$key] = $fifoLayer->only('id');
            $result[$key]['archivable'] = $archivable;
            $result[$key]['reason'] = $archivable ? null : [
                'unfulfilledQuantity' => __('messages.fifo_layer.archive_not_fully_fulfilled', [
                    'resource' => $this->resource,
                    'id' => $fifoLayer->id,
                ]),
            ];
        }

        return $this->response->addData($result);
    }

    public function inventoryMovements($fifoLayerId): Response
    {
        $fifoLayer = FifoLayer::with([])->findOrFail(e($fifoLayerId));
        $fifoLayer->inventoryMovements->load('warehouse', 'product');

        return $this->response->addData(InventoryMovementResource::collection($fifoLayer->inventoryMovements));
    }

    public function recalculateTotalCosts(Request $request)
    {
        $inputs = $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:fifo_layers,id',
        ]);

        $this->fifoLayerRepository->recalculateTotalCosts($inputs['ids']);

        return $this->response->success();
    }

    protected function getModel(): string
    {
        return FifoLayer::class;
    }

    protected function getResource(): string
    {
        return FifoLayerResource::class;
    }
}
