<?php

namespace App\Http\Controllers;

use App\DataTable\Exports\DataTableExporter;
use App\Exceptions\ExportingEmptyDataException;
use App\Response;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class ExportController extends Controller
{
    public function download(Request $request): BinaryFileResponse
    {
        $filename = $request->query('file');
        $resource = $request->query('resource');
        if (! $filename) {
            abort(404);
        }

        $disk = Storage::disk('model-exports');
        $path = $disk->path(e($filename));
        if (! $path) {
            abort(404);
        }

        return response()
            ->download($path, $filename, [
                'Content-Length' => $disk->size($filename),
                'Content-Type' => 'text/csv',
                'Content-Description' => 'File Transfer',
            ]);
    }

    /**
     * @throws ExportingEmptyDataException
     */
    public function createDownloadFile(Request $request): Response
    {
        $request->validate([
            'name' => 'required|string',
            'data' => 'required|array',
        ]);
        $data = $request->data;
        $columns = (array_keys($data[0]));
        $exporter = new DataTableExporter(
            $data,
            $request->name,
            'csv',
            null,
            array_combine($columns, $columns),
            $columns
        );

        $exporter->export();

        return $this->response
            ->addData([
                'file' => $exporter->getFilename(),
                'resource' => $request->name,
            ]);
    }
}
