<?php

namespace App\Http\Controllers;

use App\Models\UserConfig;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

/**
 * Class DBSchemaController.
 */
class DBSchemaController extends Controller
{
    public function schema(Request $request, Response $response): BinaryFileResponse
    {
        $dir = dirname(dirname(dirname(__DIR__))).'/database/schema/';
        $filename = $dir.'schema.json';

        return response()->file($filename);
    }

    public function custom(Request $request, Response $response)
    {
        $base_key = 'db_schema_custom';
        $version = $request->input('v');
        $dir = dirname(dirname(dirname(__DIR__))).'/database/schema/';
        $filename = $dir.'schema_custom.json';

        return $this->getRecordOrFile($base_key, $version, $filename);
    }

    public function meta(Request $request, Response $response)
    {
        $base_key = 'db_schema_meta';
        $version = $request->input('v');
        $dir = dirname(dirname(dirname(__DIR__))).'/database/schema/';
        $filename = $dir.'schema_meta.json';

        return $this->getRecordOrFile($base_key, $version, $filename);
    }

    protected function getRecordOrFile($base_key, $version, $filename)
    {
        if ($version === 'base') {
            // get base version from files
            return response()->file($filename);
        }

        $record = false;

        if ($version) {
            // get requested version from DB
            $record = UserConfig::where('key', $base_key.'_'.$version)->whereNull('user_id')->orderBy('updated_at', 'desc')->first();
        } else {
            // get latest version from DB
            $record = UserConfig::where('key', 'like', $base_key.'_%')->whereNull('user_id')->orderBy('updated_at', 'desc')->first();
        }

        if ($record) {
            return response()->json(json_decode($record->value));
        } else {
            // get base version from files
            return response()->file($filename);
        }
    }

    public function save(Request $request, Response $response): JsonResponse
    {
        $meta = json_encode($request->input('meta'), JSON_PRETTY_PRINT);
        $custom = json_encode($request->input('custom'), JSON_PRETTY_PRINT);

        $version = substr(md5(microtime().rand(0, 10000000)), 0, 6);

        if ($meta) {
            $this->saveRecord('db_schema_meta', $version, $meta);
        }

        if ($custom) {
            $this->saveRecord('db_schema_custom', $version, $custom);
        }

        return response()->json(['version' => $version]);
    }

    protected function saveRecord($key, $version, $data)
    {
        $record = new UserConfig();
        $record->key = $key.'_'.$version;
        $record->value = $data;

        return $record->save();
    }
}
