<?php

namespace App\Http\Controllers;

use App\Http\Requests\AllocateCustomerCreditToOrdersRequest;
use App\Http\Resources\CustomerOpenSalesCreditResource;
use App\Http\Resources\CustomerUnpaidOrderResource;
use App\Models\Customer;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Response;
use App\SalesCreditAllocation;
use Illuminate\Support\Facades\DB;

/**
 * Class CustomerAccountingController.
 */
class CustomerAccountingController extends Controller
{
    /**
     * Gets the customer's unpaid sales orders.
     */
    public function unpaidOrders($customerId): Response
    {
        $customer = Customer::with(['salesOrders'])->findOrFail(e($customerId));

        return $this->response->addData(
            CustomerUnpaidOrderResource::collection(
                $customer->salesOrders()
                    ->where('payment_status', '!=', SalesOrder::PAYMENT_STATUS_PAID)
                    ->get()->filter(function (SalesOrder $order) {
                        return $order->total - $order->total_paid > 0;
                    })
            )
        )->success();
    }

    /**
     * Gets the customer's open sales credits.
     */
    public function openSalesCredits($customerId): Response
    {
        $customer = Customer::with(['salesCredits'])->findOrFail(e($customerId));

        return $this->response->addData(
            CustomerOpenSalesCreditResource::collection(
                $customer->salesCredits()->where('credit_status', SalesCredit::CREDIT_STATUS_OPEN)->get()
            )
        );
    }

    /**
     * @return mixed
     */
    public function allocateCreditToOrders(AllocateCustomerCreditToOrdersRequest $request, $customerId, $salesCreditId)
    {
        $payload = $request->validated();

        return DB::transaction(function () use ($payload, $customerId, $salesCreditId) {
            $customer = Customer::with(['salesCredits'])->findOrFail(e($customerId));
            /** @var SalesCredit $salesCredit */
            $salesCredit = $customer->salesCredits()->with(['salesCreditLines'])->findOrFail(e($salesCreditId));
            $salesCreditPaymentType = PaymentType::getForSalesCredit();

            foreach ($payload['sales_orders'] as $order) {
                $salesOrder = SalesOrder::with([])->findOrFail(e($order['id']));

                /** @var Payment $payment */
                $payment = $salesOrder->payments()->create([
                    'payment_date' => now(),
                    'payment_type_id' => $salesCreditPaymentType->id,
                    'amount' => $order['amount'],
                ]);

                SalesCreditAllocation::with([])->create([
                    'sales_credit_id' => $salesCreditId,
                    'sales_order_id' => $salesOrder->id,
                    'amount' => $order['amount'],
                ]);

                // Set the sales credit payment status
                $salesCredit->paid($payment->payment_date);
            }

            return $this->response->setMessage(__('messages.success.create', ['resource' => 'sales credit allocation']));
        });
    }
}
