<?php

namespace App\Http\Controllers;

use App\Http\Requests\BulkUpdateCurrencies;
use App\Http\Requests\StoreCurrency;
use App\Http\Resources\CurrencyResource;
use App\Jobs\UpdateDefaultCurrencyJob;
use App\Models\Currency;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CurrencyController extends Controller
{
    private $resource = 'currency';

    /**
     * Display a listing of the currency.
     */
    public function index(): JsonResponse
    {
        return $this->response->addData(CurrencyResource::collection(Currency::with(['constantCurrency'])->orderBy('is_default', 'desc')->get()));
    }

    /**
     * Store a newly created currency.
     */
    public function store(StoreCurrency $request): JsonResponse
    {
        $currency = new Currency($request->validated());
        $currency->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData($currency);
    }

    /**
     * Set the specified currency as default currency.
     */
    public function setAsDefault(Currency $currency): JsonResponse
    {
        $currency->is_default = true;
        $currency->save();

        return $this->response
            ->setMessage(__('messages.success.default', [
                'resource' => $this->resource,
                'id' => $currency->code,
            ]))
            ->addData($currency);
    }

    /**
     * Update the specified currency.
     */
    public function update(StoreCurrency $request, $id): JsonResponse
    {
        $currency = Currency::with([])->findOrFail($id);
        $currency->fill($request->validated());
        $currency->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $currency->code,
            ]))
            ->addData($currency);
    }

    /**
     * Bulk update currencies.
     */
    public function bulkUpdate(BulkUpdateCurrencies $request): Response
    {
        $defaultCurrencyChanged = false;
        $oldDefaultCurrency = Currency::default();
        foreach (collect($request->input('currencies', []))->sortByDesc('is_default') as $currencyInput) {
            $currency = Currency::with([])->findOrFail($currencyInput['id']);
            $currency->conversion = $currencyInput['conversion'];
            $currency->is_default = $currencyInput['is_default'] ?? $currency->is_default;
            $currency->save();
            if ($currency->is_default && $currency->id != $oldDefaultCurrency->id) {
                $defaultCurrencyChanged = true;
            }
        }

        // update the reporting and others after update all currencies to use the new conversion rate to the previous default currency
        if ($defaultCurrencyChanged) {
            UpdateDefaultCurrencyJob::dispatchSync($oldDefaultCurrency->id);
        }

        return $this->response->setMessage(__('messages.success.bulk_update', ['resources' => 'currencies']));
    }

    /**
     * Remove the specified currency.
     *
     *
     * @throws Exception
     */
    public function destroy(Currency $currency): JsonResponse
    {
        $reasons = $currency->delete();

        // check if the nominalCode is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['currency_id' => $currency->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'currency',
                    'id' => $currency->code,
                ]));
        }

        return $this->response->error(Response::HTTP_BAD_REQUEST)
            ->addError(__('messages.currency.delete_failed'), 'CurrencyId'.Response::CODE_UNACCEPTABLE, 'id', ['id' => $currency->id]);
    }

    /**
     * Bulk delete currencies.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:currencies,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $currencies = Currency::with([])->whereIn('id', $ids)->get();
        foreach ($currencies as $currency) {
            $reasons = $currency->delete();

            // check if the resource is linked
            if ($reasons and is_array($reasons)) {
                foreach ($reasons as $key => $reason) {
                    $this->response->addWarning($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, "ids.{$currency->id}.{$key}");
                }
            }
        }

        return $this->response->setMessage(__('messages.currency.bulk_delete_success'));
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:currencies,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $currencies = Currency::with([])->whereIn('id', $ids)->get();
        foreach ($currencies as $key => $currency) {
            $isUsed = $currency->isUsed();

            $result[$key] = $currency->only('id', 'is_default', 'code');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }
}
