<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Resources\CsvTemplateResource;
use App\Models\Attribute;
use App\Models\CsvTemplate;
use App\Models\ProductPricingTier;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Response;
use DirectoryIterator;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class CsvTemplateController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = CsvTemplate::class;

    private $importTmpDir = '/sku'; // temp directory

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): Response
    {
        $request->validate([
            'name' => [
                'required',
                'max:255',
                Rule::unique('csv_templates', 'name')->where('model', $request->input('model')),
            ],
            'model' => 'required|in:'.implode(',', CsvTemplate::MODELS),
            'columns' => 'required|array|min:1',
        ]);

        $template = CsvTemplate::with([])->create($request->all());

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'csv template']))
            ->addData($template);
    }

    /**
     * Display the specified resource.
     */
    public function show(CsvTemplate $csvTemplate): Response
    {
        return $this->response->addData($csvTemplate);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, CsvTemplate $csvTemplate): Response
    {
        $request->validate([
            'name' => [
                'required',
                'max:255',
                Rule::unique('csv_templates', 'name')
                    ->where('model', $request->input('model'))
                    ->ignoreModel($csvTemplate),
            ],
            'model' => 'required|in:'.implode(',', CsvTemplate::MODELS),
            'columns' => 'required|array|min:1',
        ]);

        $csvTemplate->fill($request->all());
        $csvTemplate->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => 'csv template',
                'id' => $csvTemplate->name,
            ]))
            ->addData($csvTemplate);
    }

    /**
     * Remove the specified resource from storage.
     *
     *
     * @throws Exception
     */
    public function destroy(CsvTemplate $csvTemplate): Response
    {
        $csvTemplate->delete();

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'csv template',
            'id' => $csvTemplate->name,
        ]));
    }

    /**
     * Return url file of sample template.
     *
     *
     * @return string
     */
    public function sample(CsvTemplate $csvTemplate)
    {
        return $csvTemplate->sample;
    }

    /**
     * Show the Required data for editing/creating a new template.
     *
     *
     * @throws ValidationException
     */
    public function prepare(string $model): Response
    {
        Validator::make(['model' => $model], ['model' => 'in:'.implode(',', CsvTemplate::MODELS)])->validate();

        if ($model === CsvTemplate::MODEL_PRODUCT) {
            return $this->response->addData([
                'attributes' => [
                    'id',
                    'sku',
                    'name',
                    'barcode',
                    'type',
                    'weight',
                    'weight_unit',
                    'brand',
                    'tags',
                    'image_url',
                    'category',
                    'category_others',
                    'length',
                    'width',
                    'height',
                    'dimension_unit',
                    'fba_prep_instructions',
                    'case_quantity',
                    'case_length',
                    'case_width',
                    'case_height',
                    'case_dimension_unit',
                    'case_weight',
                    'case_weight_unit',
                    'average_cost',
                    'created_at',
                ],
                'relations' => [
                    'productAttributes' => Attribute::all(['id', 'name']),
                    'productPricingTiers' => ProductPricingTier::all(['id', 'name']),
                    'suppliers' => Supplier::all(['id', 'name']),
                    'SupplierPricingTiers' => SupplierPricingTier::all(['id', 'name']),
                ],
            ]);
        } elseif ($model == CsvTemplate::MODEL_SALES_ORDER) {
            return $this->response->addData([
                'attributes' => [
                    'sales_order_number',
                    'sales_channel',
                    'customer',
                    'order_date',
                    'ship_by',
                    'deliver_by',
                    'shipping_method',
                    'warehouse_name',
                    'items',
                ],
            ]);
        } elseif ($model == CsvTemplate::MODEL_PRODUCT_BRAND) {
            return $this->response->addData([
                'attributes' => [
                    'id',
                    'name',
                ],
            ]);
        } elseif ($model == CsvTemplate::MODEL_SUPPLIER) {
            return $this->response->addData([
                'attributes' => [
                    'id',
                    'name',
                    'company_name',
                    'email',
                    'phone',
                    'primary_contact_name',
                    'purchase_order_email',
                    'website',
                    'leadtime',
                    'minimum_order_quantity',
                    'minimum_purchase_order',
                    'company',
                    'address1',
                    'address2',
                    'address3',
                    'city',
                    'province',
                    'province_code',
                    'zip',
                    'country',
                    'country_code',
                ],
            ]);
        }
    }

    /**
     * Import CSV file based on template.
     *
     *
     * @return Response|array
     *
     * @throws Exception
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required_without:tmp_file|file',
            'tmp_file' => 'required_without:file',
            'template_id' => 'required_without:model|exists:csv_templates,id',
            'model' => 'required_without:template_id|in:'.implode(CsvTemplate::MODELS),
            'preview' => 'boolean',
        ]);

        $preview = $request->input('preview', false);

        $this->removeOldTmpFiles(); // delete old temp files

        // fil path from file or tmp_file
        if ($request->hasFile('file')) {
            $file = $request->file('file');
            // add file to tmp files if preview
            if ($preview) {
                $file = $file->move(sys_get_temp_dir().$this->importTmpDir);

                $this->response->addData($file->getRealPath(), 'tmp_file');
            }

            $filePath = $file->getRealPath();
        } elseif (file_exists($request->input('tmp_file'))) {
            $filePath = $request->input('tmp_file');
        } else {
            // temp file not exists, it is old file (created before 1 hour)
            $this->response->addError(__('messages.import_export.tmp_file_not_exists', ['input' => $request->input('tmp_file')]), Response::CODE_FILE_NOT_FOUND, 'tmp_file', ['tmp_file' => $request->input('tmp_file')]);

            return $this->response;
        }

        $selectedTemplate = null;
        if (! empty($request->input('template_id'))) {
            $selectedTemplate = CsvTemplate::with([])->find($request->input('template_id'));
        } else {
            // auto detect a template by the file columns
            foreach (CsvTemplate::with([])->where('model', $request->input('model'))->get() as $csvTemplate) {
                if ($csvTemplate->isSame($filePath)) {
                    $selectedTemplate = $csvTemplate;
                    $this->response->addData([
                        'id' => $selectedTemplate->id,
                        'name' => $selectedTemplate->name,
                    ], 'selected_template');
                    break;
                }
            }
        }

        if ($selectedTemplate) {
            // import or preview 10 rows
            $this->response = $selectedTemplate->import($filePath, $preview, $preview ? 10 : null);
        } else {
            // can't detect matching between file columns and templates
            $this->response->addError(__('messages.import_export.mismatch_template'), Response::CODE_INVALID_Template, 'file');
        }

        return $this->response;
    }

    /**
     * @return array|string
     *
     * @throws Exception
     */
    public function export(Request $request, CsvTemplate $csvTemplate)
    {
        $callable = function () use ($request, $csvTemplate) {
            $builder = $csvTemplate->getModelBuilder();

            $builder->filter();
            $builder->sort();
            $builder->archived($request->get('archived', 0));

            $page = 1;
            do {
                $models = $builder->forPage($page++)->get();

                if ($csvTemplate->model == CsvTemplate::MODEL_SALES_ORDER) {
                    foreach ($models as $salesOrder) {
                        foreach ($salesOrder->salesOrderLines as $salesOrderLine) {
                            $salesOrderLine->setRelation('salesOrder', $salesOrder);

                            yield $salesOrderLine;
                        }
                    }
                } else {
                    yield $models;
                }
            } while ($models->count());
        };

        return $csvTemplate->export(false, $request->get('limit'), $callable);
    }

    /**
     * Delete tmp files from tmp directory.
     *
     * delete files that created date more that 1 hours
     */
    private function removeOldTmpFiles()
    {
        $dir = sys_get_temp_dir().$this->importTmpDir;
        $hours = 1; // hours to delete the file

        if (file_exists($dir)) {
            foreach (new DirectoryIterator($dir) as $file) {
                if ($file->isFile()) {
                    if ((time() - $file->getCTime()) > (60 * 60 * $hours)) {
                        unlink($file->getRealPath());
                    }
                }
            }
        }
    }

    /**
     * Archive the incoterm.
     */
    public function archive(CsvTemplate $csvTemplate): JsonResponse
    {
        if ($csvTemplate->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => 'csv template',
                    'id' => $csvTemplate->name,
                ]))
                ->addData(CsvTemplateResource::make($csvTemplate));
        }

        return $this->response
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => 'csv template',
                'id' => $csvTemplate->name,
            ]), 'CsvTemplate'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $csvTemplate->id])
            ->addData(CsvTemplateResource::make($csvTemplate));
    }

    /**
     * Unarchived the incoterm.
     */
    public function unarchived(CsvTemplate $csvTemplate): JsonResponse
    {
        if ($csvTemplate->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'csv template',
                    'id' => $csvTemplate->name,
                ]))
                ->addData(CsvTemplateResource::make($csvTemplate));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'csv template',
                'id' => $csvTemplate->name,
            ]), 'CsvTemplate'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $csvTemplate->id])
            ->addData(CsvTemplateResource::make($csvTemplate));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return CsvTemplate::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return CsvTemplateResource::class;
    }
}
