<?php

namespace App\Http\Controllers;

use App\Actions\ReallocateBackorderRelease;
use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Resources\BackorderQueueReleaseResource;
use App\Http\Resources\BackorderQueueResource;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueRelease;
use App\Models\SalesOrder;
use App\Repositories\BackorderQueueRepository;
use App\Response;
use App\Services\InventoryManagement\BackorderManager;
use App\Services\SalesOrder\Backorder\PriorityOrderFactory;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use Throwable;

/**
 * Class BackorderQueueController.
 */
class BackorderQueueController extends Controller
{
    use BulkOperation;
    use DataTable
  {
      index as traitIndex;
  }

    /**
     * @var BackorderQueueRepository
     */
    protected $queues;

    /**
     * @var string
     */
    protected $model_path = BackorderQueue::class;

    /**
     * @var string
     */
    protected $resource = 'backorder queue';

    /**
     * BackorderQueueController constructor.
     */
    public function __construct(BackorderQueueRepository $queues)
    {
        parent::__construct();
        $this->queues = $queues;
    }

    /**
     * @return JsonResponse|AnonymousResourceCollection
     */
    public function index(Request $request)
    {
        // only return "table_specifications" if its input is "1"
        if ($this->shouldReturnTableSpecificationsOnly($request)) {
            return $this->getResponseWithTableSpecifications();
        }

        // prevent send included and excluded together
        if (! $this->validateIncludedExcluded($request)) {
            return $this->getResponseWithIncludedExcludedError();
        }

        $builder = $this->buildIndexQuery($request);
        $builder->whereColumn('backordered_quantity', '>', 'released_quantity');
        //        $builder->where(function (Builder $builder) {
        //            $builder->where('priority', '!=', 0)
        //              ->whereNotNull('priority');
        //        });

        $builder = DataTableConfiguration::paginate($builder);

        return $this->getResource()::collectionWithTableSpecifications($builder, $this->getModel());
    }

    public function show(BackorderQueue $backorderQueue): Response
    {
        $backorderQueue->load(DataTableConfiguration::getRequiredRelations($this->model_path));

        return $this->response->addData(BackorderQueueResource::make($backorderQueue));
    }

    /**
     * @return BackorderQueueResource|Response
     *
     * @throws BindingResolutionException
     */
    public function reorder(Request $request, BackorderQueue $backorderQueue)
    {
        $request->validate([
            'reorder_type' => 'required|in:'.implode(',', BackorderQueue::REORDER_TYPES),
            'options' => 'nullable|array',
        ]);

        try {
            return $this->response->addData(
                BackorderQueueResource::make(
                    PriorityOrderFactory::make(
                        $request->input('reorder_type'),
                        $backorderQueue,
                        $request->get('options', [])
                    )->process()
                )
            );
        } catch (InvalidArgumentException $e) {
            return $this->response->addError($e->getMessage(), 'InvalidArgument', $backorderQueue->id);
        }
    }

    /**
     * @throws Throwable
     */
    public function reallocateRelease(Request $request, BackorderQueueRelease $backorderQueueRelease): ?Response
    {
        try {
            $request->validate([
                'sales_order_id' => 'required|exists:sales_orders,id',
                'quantity' => 'numeric|gt:0',
            ]);

            $salesOrder = SalesOrder::findOrFail($request->input('sales_order_id'));

            $releases = (new ReallocateBackorderRelease(
                $request->input('quantity'),
                $backorderQueueRelease,
                $salesOrder
            ))->handle();

            return $this->response->addData(BackorderQueueReleaseResource::collection($releases));
        } catch (Exception $e) {
            return $this->response->addError($e->getMessage(), 'BackorderQueueReleaseAllocationError', $backorderQueueRelease->id)->setStatusCode(500);
        }
    }

    /**
     * @throws BindingResolutionException
     */
    public function reorderBulk(Request $request): Response
    {
        $request->validate([
            'reorder_type' => 'required|in:'.implode(',', BackorderQueue::REORDER_TYPES),
            'options' => 'nullable|array',
            'options.ids' => 'required|array',
            'options.ids.*' => 'exists:backorder_queues,id',
            'options.reference_id' => 'required_with:options.ids.*|exists:backorder_queues,id',
        ]);

        DB::beginTransaction();
        try {
            PriorityOrderFactory::make(
                $request->input('reorder_type'),
                null,
                $request->get('options', [])
            )->process();
            DB::commit();

            return $this->response->addData('Reordering completed successfully.');
        } catch (InvalidArgumentException $e) {
            DB::rollBack();

            return $this->response->addError($e->getMessage(), 'InvalidArgument', '');
        }
    }

    public function switchSupplier(Request $request, BackorderQueue $backorderQueue): Response
    {
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
        ]);

        try {
            return $this->response->addData(
                BackorderQueueResource::make(
                    $this->queues->switchSupplier($backorderQueue, $request->get('supplier_id'))
                )
            );
        } catch (InvalidArgumentException $e) {
            return $this->response->addError($e->getMessage(), 'SupplierId', 'SupplierId');
        }
    }

    public function switchSuppliers(Request $request)
    {
        $request->validate([
            'queues' => 'required|array',
            'queues.*.supplier_id' => 'required|exists:suppliers,id',
            'queues.*.queue_id' => 'required|exists:backorder_queues,id',
        ]);

        $errors = $this->queues->switchSuppliers($request->get('queues'));
        if (empty($errors)) {
            return $this->response->addData('Suppliers successfully switched.');
        }

        return $this->response->addWarning(
            'Some suppliers could not be switched',
            'SwitchSupplier',
            'SupplierId',
            $errors
        );
    }

    public function supplierAvailability(Request $request): Response
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'required|exists:backorder_queues,id',
        ]);

        return $this->response->addData(
            $this->queues->supplierAvailabilityForQueues($request->get('ids'))
        );
    }

    public function history(BackorderQueue $backorderQueue): Response
    {
        return $this->response->addData(
            app(BackorderManager::class)->getHistoryAndNextScheduledPurchaseOrder($backorderQueue)
        );
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    public function archive(BackorderQueue $backorderQueue): Response
    {
        if ($backorderQueue->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $backorderQueue->id,
                ]))
                ->addData(BackorderQueueResource::make($backorderQueue));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $backorderQueue->id,
            ]), 'BackorderQueue'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $backorderQueue->id])
            ->addData(BackorderQueueResource::make($backorderQueue));
    }

    public function unarchived(BackorderQueue $backorderQueue): Response
    {
        if ($backorderQueue->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $backorderQueue->id,
                ]))
                ->addData(BackorderQueueResource::make($backorderQueue));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $backorderQueue->id,
            ]), 'BackorderQueue'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $backorderQueue->id])
            ->addData(BackorderQueueResource::make($backorderQueue));
    }

    protected function getModel(): string
    {
        return BackorderQueue::class;
    }

    protected function getResource(): string
    {
        return BackorderQueueResource::class;
    }
}
