<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/27/20
 * Time: 10:16 AM.
 */

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Resources\BackorderLayerResource;
use App\Models\BackorderLayer;
use App\Response;
use Exception;
use Illuminate\Http\Request;

class BackorderLayerController extends Controller
{
    use BulkOperation, DataTable;

    /**
     * @var string
     */
    protected $model_path = BackorderLayer::class;

    /**
     * @var string
     */
    protected $resource = 'backorder layer';

    public function show(BackorderLayer $backorderLayer): Response
    {
        $backorderLayer->load(DataTableConfiguration::getRequiredRelations($this->model_path));

        return $this->response->addData(BackorderLayerResource::make($backorderLayer));
    }

    /**
     * @return $this|Response
     */
    public function fetchForProduct(Request $request, $productId)
    {
        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications(BackorderLayer::class));
        }

        // prevent send included and excluded together
        if ($request->has('included') and $request->has('excluded')) {
            return $this->response->error(Response::HTTP_BAD_REQUEST, [__('messages.failed.not_both_include_exclude')])
                ->setMessage(__('messages.failed.not_both_include_exclude'));
        }

        // set default included
        $this->setDefaultIncludedColumns($request, BackorderLayer::class);

        $backorderLayers = BackorderLayer::with([])->where('backorder_layers.product_id', e($productId));

        $backorderLayers->filter($request);
        $backorderLayers->addRelations($request);
        $backorderLayers->addCustomColumns();
        $backorderLayers->sort($request);

        // paginate with limit per page (default 10)
        $backorderLayers = DataTableConfiguration::paginate($backorderLayers);

        return $this->response->addData(BackorderLayerResource::collectionWithTableSpecifications($backorderLayers, BackorderLayer::class));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    public function archive(BackorderLayer $backorderLayer): Response
    {
        if ($backorderLayer->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $backorderLayer->id,
                ]))
                ->addData(BackorderLayerResource::make($backorderLayer));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $backorderLayer->id,
            ]), 'BackorderLayer'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $backorderLayer->id])
            ->addData(BackorderLayerResource::make($backorderLayer));
    }

    public function unarchived(BackorderLayer $backorderLayer): Response
    {
        if ($backorderLayer->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $backorderLayer->id,
                ]))
                ->addData(BackorderLayerResource::make($backorderLayer));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $backorderLayer->id,
            ]), 'BackorderLayer'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $backorderLayer->id])
            ->addData(BackorderLayerResource::make($backorderLayer));
    }

    protected function getModel(): string
    {
        return BackorderLayer::class;
    }

    protected function getResource(): string
    {
        return BackorderLayerResource::class;
    }
}
