<?php

namespace App\Http\Controllers;

use App\Http\Requests\Auth\UpdatePasswordRequest;
use App\Http\Requests\Auth\UpdateProfileRequest;
use App\Http\Resources\UserResource;
use App\Mail\Auth2FAToken;
use App\Mail\PasswordUpdatedMail;
use App\Models\User;
use App\Models\UserAlert;
use App\Response;
use Carbon\Carbon;
use Exception;
use Illuminate\Cache\ArrayStore;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Laravel\Sanctum\NewAccessToken;

class AuthController extends Controller
{
    use AuthenticatesUsers;

    protected $maxAttempts = 5; // Amount of bad attempts user can make

    protected $decayMinutes = 1; // Time for which user is going to be blocked in minutes

    /**
     * Get a JWT via given credentials.
     *
     *
     * @throws ValidationException
     */
    public function login(Request $request): JsonResponse
    {
        // header("X-received-csrf: ".$request->csrf);
        // filter out everything that is not username (email in our case) and password
        // $credentials = $this->credentials($request);

        if (env('AUTH_2FA_ENABLED', false)) {
            $alleged_user = $this->getUserWithCredentials($this->credentials($request));
            if ($alleged_user &&
                $alleged_user->requires2FA()) {
                if (empty($request->email2fa) && ! empty($request->new2fa)) {
                    $this->send2FAEmail($alleged_user);

                    return $this->askFor2FA();
                } elseif (! $alleged_user->is2FATokenValid($request->email2fa)) {
                    return $this->reject2FA();
                }
            }
        }

        // if ((! $request->email || ! $request->password) && Auth::id()) {
        //   // No credentials provided.  Use logged user.
        //   header("X-ckpt1: 1");
        //   return $this->respondWithToken(auth()->tokenById(Auth::id()));
        // }

        // If credentials match, then these two are equivalent:
        // dd( auth()->tokenById(Auth::id()) );
        // dd( auth()->attempt( ['email'=>'admin@...', 'password'=>'...secret'] ) );

        // header("X-Session-Lifetime: ".\Config::get('session.lifetime'));

        // header("X-ckpt2: 2");
        // $request->validate([
        //   'email'    => 'required',
        //   'password' => 'required',
        // ]);

        // header("X-ckpt3: 3");
        // $credentials = $request->only(['email', 'password']);

        //

        // If the class is using the ThrottlesLogins trait, we can automatically throttle
        // the login attempts for this application. We'll key this by the username and
        // the IP address of the client making these requests into this application.
        if (method_exists($this, 'hasTooManyLoginAttempts') &&
            $this->hasTooManyLoginAttempts($request)) {
            // header("X-ckpt4: 4");
            $this->fireLockoutEvent($request);

            return $this->sendLockoutResponse($request);
        }

        if ($this->attemptLogin($request))
        {
            if (auth()->user()->deleted_at) {
                return response()->json(['error' => 'Unauthorized'], 401);
            }

            if (! $token = auth()->user()->createToken('API Token')) {
                // If the login attempt was unsuccessful we will increment the number of attempts
                // to login and redirect the user back to the login form. Of course, when this
                // user surpasses their maximum number of attempts they will get locked out.
                // header("X-ckpt5: 5");
                $this->incrementLoginAttempts($request);

                return response()->json(['error' => 'Unauthorized'], 401);
            }
        } else {
            return response()->json(['error' => 'Credentials do not match'], 401);
        }

        // header("X-ckpt6: 6");
        auth()->user()->update(['last_login_at' => now()]);
        // an also respond with token
        return $this->respondWithToken($token);
    }

    protected function send2FAEmail($alleged_user)
    {
        $token = $alleged_user->generate2FAToken();
        $alleged_user->save();
        \Mail::to($alleged_user)->queue(new Auth2FAToken($alleged_user));
    }

    protected function askFor2FA()
    {
        return response()->json(['error' => '2fa_email'], 401);
    }

    protected function reject2FA()
    {
        return response()->json(['error' => 'The 2FA token you provided is invalid'], 401);
    }

    protected function getUserWithCredentials($credentials)
    {
        $user = User::where(
            [
                $this->username() => $credentials[$this->username()],
            ]
        )->first();
        $user = password_verify($credentials['password'], $user?->getAuthPassword()) ?
            $user : false;

        return $user;
    }

    /**
     * Get the authenticated User.
     */
    public function profile(): JsonResponse
    {
        return $this->response->addData(UserResource::make(auth()->user()));
    }

    /**
     * Log the user out (Invalidate the token).
     *
     * @throws Exception
     */
    public function deleteAccessToken(): JsonResponse
    {
        /**
         * Check if cache driver is array.
         */
        if (cache()->getStore() instanceof ArrayStore) {
            return response()->json(['error' => 'This is not available when CACHE_DRIVER is array.'], 501);
        }

        auth()->user()->tokens()->delete();

        return response()->json(['message' => 'API access token deleted successfully.']);
    }

    /**
     * Refresh a token.
     */
    public function refresh(): JsonResponse
    {
        try {
            return $this->respondWithToken(auth()->refresh());
        } catch (\Throwable $exception) {
            return response()->json(['status' => 'failed', 'message' => $exception->getMessage()], Response::HTTP_BAD_REQUEST);
        }
    }

    /**
     * Retrieve user alerts.
     *
     *
     * @return mixed
     */
    public function alerts(Request $request)
    {
        $alerts = auth()->user()->alerts();

        return $alerts->paginate($request->get('limit', 10));
    }

    /**
     * Retrieve unread notifications.
     *
     *
     * @return mixed
     */
    public function unreadNotifications(Request $request)
    {
        $notifications = auth()->user()->alerts()->where(function ($query) {
            $query->where('show_in_ui_notification', 1)
                ->whereNull('read_at');
        });

        return $notifications->paginate($request->get('limit', 10));
    }

    /**
     * Mark notification as seen.
     *
     *
     * @return array
     */
    public function readNotification($activity_log_id)
    {
        $alert = UserAlert::with([])->where([
            'user_id' => auth()->id(),
            'activity_log_id' => $activity_log_id,
        ])->firstOrFail();
        if ($alert) {
            $alert->read_at = Carbon::now();
            $alert->save();
        }

        return ['status' => 'success'];
    }

    /**
     * Get the token array structure.
     */
    protected function respondWithToken(NewAccessToken $token): JsonResponse
    {
        return response()->json([
            'access_token' => $token->plainTextToken,
            'token_type' => 'bearer',
        ]);
    }

    /**
     * Get the login username to be used by the controller.
     *
     * @return string
     */
    public function username()
    {
        return 'email';
    }

    public function updateProfile(UpdateProfileRequest $request): JsonResponse
    {
        $user = auth()->user();
        $user->fill($request->validated());
        $user->update();

        return response()->json(['message' => 'Successfully updated']);
    }

    public function updatePassword(UpdatePasswordRequest $request): JsonResponse
    {
        $user = auth()->user();

        $user->update([
            'password' => bcrypt($request->password),
        ]);

        \Mail::to($user)->queue(new PasswordUpdatedMail($user));

        return response()->json(['message' => 'Successfully updated password']);
    }
}
