<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreAddressRequest;
use App\Models\Address;
use App\Response;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class AddressController extends Controller
{
    /**
     * update address by id.
     */
    public function update(StoreAddressRequest $request, Address $address): Response
    {
        if ($address->isLocked()) {
            return $this->response->addError(__('messages.address.locked'), 'Address'.Response::CODE_IS_LOCKED, 'id', ['address_id' => $address->id]);
        }

        $address->fill($request->validated())->save();

        return $this->response->addData($address->attributesToArray());
    }

    public function destroy(Address $address)
    {
        $reasons = $address->delete();

        // check if the address is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['address_id' => $address->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'address',
                    'id' => $address->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'address',
            'id' => $address->name,
        ]));
    }

    /**
     * check the possibility of deletion.
     */
    public function isLocked(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:addresses,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $addresses = Address::with([])->whereIn('id', $ids)->get();
        foreach ($addresses as $key => $address) {
            $result[$key] = $address->only('id');
            $result[$key]['locked'] = $address->isLocked();
        }

        return $this->response->addData($result);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:addresses,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $addresses = Address::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($addresses as $key => $address) {
            $isUsed = $address->isUsed();

            $result[$key] = $address->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }
}
