<?php

namespace App\Http\Controllers;

use App\Data\AccountingBulkReplaceNominalCodesData;
use App\Data\AccountingTransactionBulkEnableSyncData;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionUpdateData;
use App\DataTable\DataTable;
use App\Http\Resources\AccountingTransactionResource;
use App\Models\AccountingTransaction;
use App\Repositories\Accounting\AccountingTransactionRepository;
use App\Repositories\IntegrationInstanceRepository;
use App\Response;
use App\Services\Accounting\AccountingTransactionManager;
use Exception;
use Illuminate\Http\Request;
use Modules\Qbo\Manager\QboManager;
use Modules\Xero\Managers\XeroManager;
use Throwable;

class AccountingTransactionController extends Controller
{
    use DataTable;

    private XeroManager $xeroManager;

    private QboManager $qboManager;

    private AccountingTransactionManager $manager;

    /**
     * @throws Exception
     */
    public function __construct(
        private readonly IntegrationInstanceRepository $integrationInstances,
        private readonly AccountingTransactionRepository $transactions,
        private readonly AccountingTransactionManager $transactionManager,
    ) {
        parent::__construct();

        $this->manager = app(AccountingTransactionManager::class);
        if ($accountingInstance = $this->integrationInstances->getAccountingInstance()) {
            if ($accountingInstance->isQbo()) {
                $this->qboManager = new QboManager($accountingInstance);
            } elseif ($accountingInstance->isXero()) {
                $this->xeroManager = app(XeroManager::class);
            }
        }
    }

    public function show($id): Response
    {
        $accountingTransaction = AccountingTransaction::query()->findOrFail($id);

        $accountingTransaction->load([
            'accountingTransactionLines',
            'accountingTransactionLines.nominalCode',
            'accountingTransactionLines.taxRate',
            'accountingIntegration',
            'parent',
            'children',
        ]);

        return $this->response->addData(new AccountingTransactionResource($accountingTransaction));
    }

    /**
     * @throws Throwable
     */
    public function store(AccountingTransactionData $data): Response
    {
        $transaction = $this->transactionManager->saveAccountingTransaction($data);
        return $this->response->addData(AccountingTransactionResource::make($transaction));
    }

    public function update(AccountingTransactionUpdateData $data, AccountingTransaction $transaction): Response
    {
        $accountingTransaction = $this->manager->updateTransaction($transaction, $data);

        return $this->response->addData(AccountingTransactionResource::make($accountingTransaction));
    }

    /**
     * @throws Exception
     */
    public function syncExternal(Request $request): Response
    {
        $request = $this->handleIdsAndFilters($request, AccountingTransaction::class);

        // We ensure that there is an accounting integration.
        if (! ($instance = $this->integrationInstances->getAccountingInstance()) || (! $instance->isXero() && ! $instance->isQbo())) {
            return $this->response
                ->addError('No Xero accounting integration found', 404, 'message')
                ->error(404);
        }

        if ($instance->isXero()) {
            $this->xeroManager->syncTransactions($request->input('ids'));
        } elseif ($instance->isQbo()) {
            $this->qboManager->syncTransactions($request->input('ids'));
        }

        return $this->response->success(200);
    }

    public function clearErrors(Request $request): Response
    {
        $request = $this->handleIdsAndFilters($request, AccountingTransaction::class);

        $this->manager->clearErrors($request->input('ids'));

        return $this->response->success(200);
    }

    /**
     * @throws Exception
     */
    public function deleteIntegrationPayments(Request $request): Response
    {
        $request = $this->handleIdsAndFilters($request, AccountingTransaction::class);

        $ids = $this->transactions->getPaymentIdsFromTransactions($request->input('ids'));

        $this->xeroManager->deletePayments($ids);

        return $this->response->success(200);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function refresh(Request $request): Response
    {
        $request = $this->handleIdsAndFilters($request, AccountingTransaction::class);
        $manager = app(AccountingTransactionManager::class);
        $manager->sync($request->input('ids'));

        return $this->response->success(200);
    }

    public function unlinkFromIntegration(Request $request): Response
    {
        $request = $this->handleIdsAndFilters($request, AccountingTransaction::class);

        $this->transactions->unLinkFromIntegration($request->input('ids'));

        return $this->response->success(200);
    }

    /**
     * @throws Throwable
     */
    public function syncAll(): Response
    {
        $this->manager->sync();

        return $this->response->success(200);
    }

    public function destroy(AccountingTransaction $transaction): Response
    {
        $transaction->delete();

        return $this->response->success(200);
    }

    public function bulkDestroy(Request $request): Response
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'required|integer',
        ]);

        AccountingTransaction::whereIn('id', $request->ids)->each(function ($transaction) {
            $transaction->delete();
        });

        return $this->response->success(200);
    }

    public function bulkReplaceNominalCodes(Request $request): Response
    {
        $request->validate([
            'ids' => 'sometimes|array|min:1',
            'ids.*' => 'integer|exists:accounting_transactions,id',
            'filters' => 'sometimes|array',
            'is_locked' => 'required|boolean',
            'old_nominal_code_id' => 'required|integer|exists:nominal_codes,id',
            'new_nominal_code_id' => 'required|integer|exists:nominal_codes,id',
        ]);

        if ($request->has('filters')) {
            $ids = $this->getIdsFromFilters(AccountingTransaction::class, $request);
        } else {
            $ids = $request->input('ids');
        }
        $this->manager->bulkReplaceNominalCodes(AccountingBulkReplaceNominalCodesData::from([
            'ids' => $ids,
            'old_nominal_code_id' => $request->input('old_nominal_code_id'),
            'new_nominal_code_id' => $request->input('new_nominal_code_id'),
            'is_locked' => $request->input('is_locked'),
        ]));

        return $this->response->success(200);
    }

    public function bulkEnableSync(Request $request): Response
    {
        $request->validate([
            'ids' => 'sometimes|array|min:1',
            'ids.*' => 'integer|exists:accounting_transactions,id',
            'filters' => 'sometimes|array',
            'status' => 'required|boolean',
        ]);

        if ($request->has('filters')) {
            $ids = $this->getIdsFromFilters(AccountingTransaction::class, $request);
        } else {
            $ids = $request->input('ids');
        }
        $this->manager->bulkEnableSync(AccountingTransactionBulkEnableSyncData::from([
            'ids' => $ids,
            'status' => $request->boolean('status'),
        ]));

        return $this->response->success(200);
    }

    /**
     * @inheritDoc
     */
    protected function getModel(): string
    {
        return AccountingTransaction::class;
    }

    /**
     * @inheritDoc
     */
    protected function getResource(): string
    {
        return AccountingTransactionResource::class;
    }
}
