<?php

namespace App\Helpers;

use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

class ImportFileHelper
{
    private $filePath;

    private $fieldMappings;

    private $temporaryTableName;

    private $isTemp = true;

    private $doubleValidations;

    private $enumValidations;

    private $requiredValidations;

    private $invalidRowsTemporaryTableName;

    private $ignoreFirstRow = false;

    private $databaseColumns;

    public function setFilePath($value)
    {
        $this->filePath = $value;
    }

    public function getFilePath()
    {
        return $this->filePath;
    }

    public function getDatabaseColumns($onlyColumns = null)
    {
        return $this->databaseColumns
            ->filter(function ($item) use ($onlyColumns) {
                if ($onlyColumns) {
                    return in_array($item, $onlyColumns);
                } else {
                    return true;
                }
            });
    }

    public function getDatabaseColumnsImplodeString($prefix = null, $onlyColumns = null)
    {
        return $this->databaseColumns
            ->filter(function ($item) use ($onlyColumns) {
                if ($onlyColumns) {
                    return in_array($item, $onlyColumns);
                } else {
                    return true;
                }
            })
            ->map(function ($item) use ($prefix) {
                return "$prefix$item";
            })
            ->implode(',');
    }

    public function getDatabaseColumnsForUpdateOperation($updateTable, $stageTable, $onlyColumns = null)
    {
        return $this->databaseColumns
            ->filter(function ($item) use ($onlyColumns) {
                if ($onlyColumns) {
                    return in_array($item, $onlyColumns);
                } else {
                    return true;
                }
            })
            ->map(function ($item) use ($updateTable, $stageTable) {
                return "$updateTable.$item = $stageTable$item";
            })
            ->implode(',');
    }

    public function setFieldMappings($value)
    {
        $this->fieldMappings = collect($value);

        $this->databaseColumns = $this->fieldMappings->map(function ($item) {
            return $item['database_column'];
        });
    }

    public function setIgnoreFirstRow($value)
    {
        $this->ignoreFirstRow = $value;
    }

    private function checkIfColumnsExists($value)
    {
        return $this->fieldMappings->filter(function ($item) use ($value) {
            return $value->contains($item['database_column']);
        })
            ->map(function ($item) {
                return $item['database_column'];
            });
    }

    public function setDoubleValidations($value)
    {
        $value = collect($value);

        $this->doubleValidations = $this->checkIfColumnsExists($value)->values();

        return $this;
    }

    public function setRequiredValidations($value)
    {
        $value = collect($value);

        $this->requiredValidations = $this->checkIfColumnsExists($value)->values();

        return $this;
    }

    public function setEnumValidations($value)
    {
        $value = collect($value);

        $this->enumValidations = $value->only($this->checkIfColumnsExists($value->keys()));

        return $this;
    }

    public function getFieldMappings()
    {
        return $this->fieldMappings;
    }

    public function getTableName()
    {
        return $this->temporaryTableName;
    }

    public function getInvalidRowsTableName()
    {
        return $this->invalidRowsTemporaryTableName;
    }

    public function generateTemporaryTableStructure()
    {
        $this->temporaryTableName = 'temporary_table_'.Str::random(10);
        $this->generateSchema();
    }

    public function generateSchema()
    {
        Schema::create($this->temporaryTableName, function (Blueprint $table) {
            if ($this->isTemp) {
                $table->temporary();
            }

            foreach ($this->fieldMappings as $column) {
                $type = @$column['type'] ? $column['type'] : 'text';
                $columnName = @$column['database_column'] ? $column['database_column'] : 'text';
                $table->$type($columnName)->nullable();
            }
        });
    }

    public function getIsTemp()
    {
        return $this->isTemp;
    }

    public function setIsTemp($value)
    {
        return $this->isTemp = $value;
    }

    public function loadData()
    {
        $ignoreClause = ($this->ignoreFirstRow ? '' : 'IGNORE 1 ROWS');

        // Dump in Table
        DB::unprepared(
            "LOAD DATA LOCAL INFILE '$this->filePath'
      INTO TABLE $this->temporaryTableName
      FIELDS TERMINATED BY ','
      LINES TERMINATED BY '\r\n'
      $ignoreClause"
        );
    }

    public function addIncrementColumn()
    {
        Schema::table($this->temporaryTableName, function (Blueprint $table) {
            $table->increments('id')->first();
        });
    }

    public function generateInvalidRowsTemporaryTable()
    {
        $this->invalidRowsTemporaryTableName = 'temporary_table_'.Str::random(10);
        $temp = $this->isTemp ? 'TEMPORARY' : '';
        DB::unprepared("CREATE $temp TABLE $this->invalidRowsTemporaryTableName AS SELECT * FROM $this->temporaryTableName limit 0");
    }

    public function executeValidations()
    {
        $query = DB::table($this->temporaryTableName);

        $query = $query->where(function ($query) {
            //Double Validations
            if ($this->doubleValidations && $this->doubleValidations->count() > 0) {
                $query = $query->orWhere(function ($query) {
                    foreach ($this->doubleValidations as $columnName) {
                        $query->orWhereRaw("$this->temporaryTableName.$columnName NOT REGEXP '[+-]?([0-9]*[.])?[0-9]+'");
                    }
                });
            }

            //Required Validations
            if ($this->requiredValidations && $this->requiredValidations->count() > 0) {
                $query = $query->orWhere(function ($query) {
                    foreach ($this->requiredValidations as $columnName) {
                        $query->orWhereRaw("($this->temporaryTableName.$columnName IS NULL OR $this->temporaryTableName.$columnName = '')");
                    }
                });
            }

            //Enum Validations
            if ($this->enumValidations && $this->enumValidations->count() > 0) {
                $query = $query->orWhere(function ($query) {
                    foreach ($this->enumValidations as $columnName => $inClause) {
                        $query = $query->orWhereRaw("$this->temporaryTableName.$columnName NOT IN ($inClause)");
                    }
                });
            }
        });

        //Insert into table
        DB::unprepared("INSERT INTO $this->invalidRowsTemporaryTableName ".$query->toSql());
    }

    public function deleteInvalidRows()
    {
        DB::unprepared("DELETE FROM $this->temporaryTableName WHERE ID IN (SELECT id FROM $this->invalidRowsTemporaryTableName)");
    }

    public function importCsv($data)
    {
        //Set File
        $this->setFilePath($data['file_path']);

        //Set Field Mappins
        if (@$data['field_mappings']) {
            $this->setFieldMappings($data['field_mappings']);
        }

        //Set Temp
        $this->setIsTemp((isset($data['is_temp']) ? $data['is_temp'] : true));

        //Generate Temporary Table
        $this->generateTemporaryTableStructure();

        //Insert Data
        $this->loadData();

        //Add increment column
        $this->addIncrementColumn();

        //Create inavalid Temporary table
        $this->generateInvalidRowsTemporaryTable();

        //Execute Validations
        $this->executeValidations();

        //Execute Validations
        $this->deleteInvalidRows();

        return $this;
    }

    public function addColumns($columns)
    {
        Schema::table($this->temporaryTableName, function (Blueprint $table) use ($columns) {
            foreach ($columns as $column) {
                $type = $column['type'];
                $columnName = $column['column'];
                $table->$type($columnName)->nullable();
            }
        });
    }

    public function renameColumns($from, $to)
    {
        Schema::table($this->temporaryTableName, function (Blueprint $table) {
            $table->renameColumn('from', 'to');
        });
    }
}
