<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 12/10/20
 * Time: 11:05 PM.
 */

namespace App\Exporters;

use App\DataTable\DataTableConfiguration;
use App\Exporters\Formatters\ExportFormatterFactory;
use App\Models\SupplierInventory;
use Illuminate\Database\Eloquent\Builder;

/**
 * Class SupplierInventoryExporter.
 */
class SupplierInventoryExporter extends BaseExporter
{
    /**
     * The supplier inventory export file prefix.
     */
    const SUPPLIER_INVENTORY_EXPORT_FILE_SUFFIX = '_supplier_inventory';

    /**
     * SupplierInventoryExporter constructor.
     * In the future, we can pass a template here
     * to be used in making the exported data.
     */
    public function __construct(Builder $builder, string $format)
    {
        parent::__construct(
            $this->makeRecords($builder),
            $format,
            $this->makeFilename($format),
            $this->makeHeaders($format)
        );
    }

    private function makeRecords(Builder $builder): array
    {
        $query = $builder->with(DataTableConfiguration::getRequiredRelations(SupplierInventory::class));

        return $query->get()->map(function (SupplierInventory $inventory) {
            return [
                'inventory_id' => $inventory->id,
                'sku' => $inventory->product ? $inventory->product->sku : '',
                'supplier_sku' => $inventory->supplier_sku,
                'warehouse' => $inventory->supplier ? $inventory->supplier->name : ''.' - '.$inventory->warehouse->name,
                'quantity' => $inventory->quantity,
                'in_stock' => $inventory->in_stock,
                'eta' => $inventory->eta,
            ];
        })
            ->toArray();
    }

    /**
     * Gets the export file name.
     */
    private function makeFilename(string $format): string
    {
        return intval(date('YmdHis').rand(0, 99999)).'_'.auth()->id().self::SUPPLIER_INVENTORY_EXPORT_FILE_SUFFIX.'.'.$format;
    }

    /**
     * Needed for exporting to excel.
     */
    private function makeHeaders(string $format): array
    {
        $headers = [
            'inventory_id' => 'string',
            'sku' => 'string',
            'supplier_sku' => 'string',
            'warehouse' => 'string',
            'quantity' => 'integer',
            'in_stock' => 'string',
            'eta' => 'date',
        ];

        return $format === ExportFormatterFactory::EXPORT_FORMAT_CSV ? array_keys($headers) : $headers;
    }
}
