<?php

namespace App\Exporters\Jasper;

use App\Helpers;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;

class PackingSlipTransformer
{
    /**
     * Sales order to packing-slip format.
     */
    public static function salesOrder(SalesOrder $salesOrder): array
    {
        return [
            'grouping_id' => $salesOrder->id,
            'order_number' => $salesOrder->sales_order_number,
            'order_date' => $salesOrder->order_date->toFormattedDateString(),
            'logo' => Helpers::getBase64OfImage($salesOrder->active_store->logo_url, 250),
            'store_address' => $salesOrder->active_store->address ? $salesOrder->active_store->address->formatForInvoice(false) : '',
            'shipping_address' => $salesOrder->shippingAddress ? $salesOrder->shippingAddress->formatForInvoice(false) : '',
            'billing_address' => $salesOrder->billingAddress ? $salesOrder->billingAddress->formatForInvoice(false) : '',
            'shipping_method' => $salesOrder->shippingMethod ? $salesOrder->shippingMethod->full_name : $salesOrder->requested_shipping_method,
            'store_name' => $salesOrder->active_store->company_name ?: $salesOrder->active_store->name,
            'shipping_address_name' => $salesOrder->customer->name,
            'billing_address_name' => $salesOrder->billingAddress->name ?? '',
            'notes' => $salesOrder->notes->implode('note', ','),
            'store_message' => $salesOrder->active_store->custom_message ?: '',
        ];
    }

    /**
     * Sales order line to packing-slip items format.
     *
     *
     * @return array array of line data array
     */
    public static function salesOrderLine(SalesOrderLine $salesOrderLine, bool $checkBackorder = true): array
    {
        if ($salesOrderLine->no_audit_trail) {
            return [];
        }
        if ($salesOrderLine->quantity == 0) {
            return [];
        }

        $lineData = [
            'fulfillable' => true,
            'sku' => $salesOrderLine->product->sku ?? $salesOrderLine->sales_channel_line_id,
            'product_image' => Helpers::getBase64OfImage($salesOrderLine->product->image ?? null, 250),
            'description' => $salesOrderLine->description,
            'qty' => $salesOrderLine->quantity,
            'barcode' => $salesOrderLine->product->barcode ?? '',
            'mpn' => $salesOrderLine->product->mpn ?? '',
        ];
        $bundle = $salesOrderLine->product->bundle ?? null;
        if ($bundle) {
            $lineData = $lineData + ['bundle' => [
                'id' => $bundle->id,
                'name' => $bundle->name,
                'sku' => $bundle->sku,
                'barcode' => $bundle->barcode, ],
            ];
        }

        // product attributes
        foreach ($salesOrderLine->product?->productAttributes->where('available_for_templates', true) ?: [] as $productAttribute) {
            $attributeName = preg_replace('/[^a-z0-9]+/', '_', strtolower($productAttribute->name));

            $lineData["product_attribute_{$attributeName}"] = $productAttribute->pivot->value ?? '';
        }

        if ($checkBackorder && $salesOrderLine->activeBackorderQueue) {
            $fulfillable = array_merge($lineData, ['fulfillable' => true, 'qty' => $salesOrderLine->quantity - $salesOrderLine->activeBackorderQueue->unreleased_quantity]);
            $unfulfillable = array_merge($lineData, ['fulfillable' => false, 'qty' => $salesOrderLine->activeBackorderQueue->unreleased_quantity]);

            return collect([$fulfillable, $unfulfillable])->where('qty', '>', 0)->values()->toArray();
        } else {
            return [$lineData];
        }
    }

    /**
     * Sales order fulfillment to packing-slip format.
     */
    public static function salesOrderFulfillment(SalesOrderFulfillment $salesOrderFulfillment): array
    {
        $fulfillmentOrderData = self::salesOrder($salesOrderFulfillment->salesOrder);

        $fulfillmentOrderData['grouping_id'] = $salesOrderFulfillment->id;
        $fulfillmentOrderData['fulfillment_number'] = $salesOrderFulfillment->fulfillment_number;
        $fulfillmentOrderData['shipping_method'] = $salesOrderFulfillment->fulfilledShippingMethod ? $salesOrderFulfillment->fulfilledShippingMethod->full_name : $salesOrderFulfillment->fulfilled_shipping_method;

        if (empty($fulfillmentOrderData['shipping_method'])) {
            $fulfillmentOrderData['shipping_method'] = $salesOrderFulfillment->requestedShippingMethod ? $salesOrderFulfillment->requestedShippingMethod->full_name : $salesOrderFulfillment->requested_shipping_method;
        }

        return $fulfillmentOrderData;
    }

    /**
     * Sales order fulfillment line to packing-slip items format.
     */
    public static function salesOrderFulfillmentLine(SalesOrderFulfillmentLine $salesOrderFulfillmentLine): array
    {
        $lineData = self::salesOrderLine($salesOrderFulfillmentLine->salesOrderLine, false);

        $lineData[0]['qty'] = $salesOrderFulfillmentLine->quantity;

        return $lineData;
    }
}
