<?php

namespace App\Exporters;

use App\Exporters\Formatters\ExportFormatterFactory;
use App\Models\InventoryMovement;
use App\Models\StockTakeItem;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;

/**
 * Class InitialInventoryExporter.
 */
class InitialInventoryExporter extends BaseExporter
{
    /**
     * The initial inventory export file prefix.
     */
    const INITIAL_INVENTORY_EXPORT_FILE_SUFFIX = '_initial_inventory';

    /**
     * InitialInventoryExporter constructor.
     * In the future, we can pass a template here
     * to be used in making the exported data.
     */
    public function __construct(Collection $stockTakeItems, private readonly Warehouse $warehouse, string $format)
    {
        parent::__construct(
            $this->makeRecords($stockTakeItems),
            $format,
            $this->makeFilename($format),
            $this->makeHeaders($format)
        );
    }

    private function makeRecords(Collection $stockTakeItems): array
    {
        return $stockTakeItems->map(function (StockTakeItem $stockTakeItem) {
            return [
                'sku' => $stockTakeItem->product->sku,
                'name' => $stockTakeItem->product->name,
                'unit_cost' => $stockTakeItem->unit_cost,
                'quantity' => $stockTakeItem->qty_counted,
                'barcode' => $stockTakeItem->product->barcode,
            ];
        })->toArray();
    }

    /**
     * Gets the export file name.
     */
    private function makeFilename(string $format): string
    {
        return intval(date('YmdHis').rand(0, 99999)).'_'.auth()->id().self::INITIAL_INVENTORY_EXPORT_FILE_SUFFIX.'_' . $this->warehouse->name . '.'.$format;
    }

    /**
     * Needed for exporting to excel.
     */
    private function makeHeaders(string $format): array
    {
        $headers = [
            'sku' => 'string',
            'name' => 'string',
            'unit_cost' => 'float',
            'quantity' => 'integer',
            'barcode' => 'string',
        ];

        return $format === ExportFormatterFactory::EXPORT_FORMAT_CSV ? array_keys($headers) : $headers;
    }
}
