<?php

namespace App\Exceptions;

use App\Helpers;
use App\Response;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Contracts\Container\Container;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\QueryException;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpKernel\Exception\MethodNotAllowedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\UnauthorizedHttpException;
use Throwable;

class Handler extends ExceptionHandler
{
    /**
     * A list of the inputs that are never flashed for validation exceptions.
     *
     * @var array
     */
    protected $dontFlash = [
        'current_password',
        'current_password',
        'password',
        'password_confirmation',
    ];

    public function __construct(Container $container)
    {
        $this->dontReport = config('sentry_custom.dont_report', []);

        return parent::__construct($container);
    }

    /**
     * Report or log an exception.
     *
     *
     * @return void
     *
     * @throws \Throwable
     */
    public function report(Throwable $exception)
    {
        if (app()->bound('sentry') && $this->shouldReport($exception) && env('APP_ENV') !== 'local') {
            $classname = get_class($exception);
            $sample_rate = config('sentry_custom.sample_rate', []);

            $rate = Arr::get($sample_rate, $classname, 1);
            $report = (rand(0, 10000) / 10000) <= $rate;

            if ($report && env('APP_ENV', 'local') !== 'local') {
                app('sentry')->captureException($exception);
            }
        }

        //SKU-6168
        if (str_contains($exception->getMessage(), 'Unknown (generic) error from engine" from storage engine')) {
            customLog('ssd_errors', $exception->getMessage(), [], 7);

            return;
        }

        // SQLSTATE[40001]: Serialization failure: 1213 Deadlock found when trying to get lock

        if ($exception instanceof QueryException && (($exception->errorInfo[0] ?? 0) == 40001 || ($exception->errorInfo[1] ?? 0) == 1637)

        ) {
            Helpers::logMysqlDeadlockException($exception);
        }

        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param  Request  $request
     * @return Response|\Illuminate\Http\Response
     */
    public function render($request, Throwable $exception)
    {
        if ($request->wantsJson()) {
            if ($exception instanceof UnauthorizedHttpException || $exception instanceof AuthenticationException) {
                return Response::make(Response::HTTP_UNAUTHORIZED)->addError(__('messages.errors.unauthorized'), Response::CODE_UNAUTHORIZED, 'token');
            }

            if ($exception instanceof AuthorizationException) {
                return Response::make(Response::HTTP_FORBIDDEN)->addError(__('messages.errors.forbidden'), Response::CODE_FORBIDDEN, 'user');
            }

            if ($exception instanceof NotFoundHttpException || $exception instanceof ModelNotFoundException) {
                $message = $exception->getMessage() ?: __('messages.errors.not_found_http');
                $key = method_exists($exception, 'getModel') ? $exception->getModel() : 'url';

                return Response::make(Response::HTTP_NOT_FOUND)->addError($message, Response::CODE_NOT_FOUND, $key, [], true);
            }

            if ($exception instanceof MethodNotAllowedHttpException) {
                return Response::make(Response::HTTP_METHOD_NOT_ALLOWED)->addError($exception->getMessage(), Response::CODE_METHOD_NOT_ALLOWED, 'method', [], true);
            }

            if (Str::endsWith($exception->getFile(), 'jwt/src/Signer/Hmac.php') and empty(env('JWT_SECRET'))) {
                return Response::make(Response::HTTP_NOT_IMPLEMENTED)
                    ->addError(__('messages.errors.jwt_key_missing'), Response::CODE_JWT_KEY_MESSING, 'JWT_SECRET');
            }

            if ($exception instanceof ValidationException) {
                return Response::make($exception->status)
                    ->setMessage($exception->getMessage())
                    ->setErrors($exception->errors());
            }

            if ($exception instanceof OpenStockTakeException) {
                return Response::make($exception->status)
                    ->setMessage($exception->getMessage())
                    ->setErrors(['resource' => 'stock take', 'id' => $exception->stockTake->id]);
            }

            if ($exception instanceof ActionUnavailableTemporaryException) {
                return Response::make($exception->getCode())
                    ->setMessage($exception->getMessage())
                    ->setErrors($exception->getResponseError('id'));
            }
        }

        return parent::render($request, $exception);
    }
}
