<?php

namespace App\DataTable\Exports;

use App\Jobs\ExportDataTable;
use App\Mail\DataTableExportMail;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\User;
use Generator;
use Illuminate\Bus\Batch;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Mail;
use Throwable;

class ExportBus
{
    const EXPORT_DATA_FETCH_PER_PAGE = 900;

    /**
     * @var string
     */
    public $fileName;

    /**
     * @var string
     */
    public $format;

    /**
     * @var User
     */
    private $user;

    /**
     * @var array The records to be exported
     */
    private $records;

    /**
     * @var array The export field mapping
     */
    private $mapping;

    /**
     * @var array Included fields
     */
    private $includedFields = [];

    /**
     * @var string
     */
    private $model;

    /**
     * @var string
     */
    private $resource;

    /**
     * @var array
     */
    private $filters = [];

    /**
     * @var array
     */
    private $sorts = [];

    /**
     * @var int
     */
    private $archived = 0;

    /**
     * Extra bindings for the query to
     * retrieve the export data.
     *
     * @var array
     */
    private $extraQueryBindings = [];

    /**
     * DataTableExportJob constructor.
     *
     * @param  User|mixed  $user
     */
    public function __construct(User $user, string $fileName, string $format, array $mapping)
    {
        $this->user = $user;
        $this->fileName = $fileName;
        $this->format = $format;
        $this->mapping = $mapping;
    }

    /**
     * Execute the job.
     */
    public function process(): void
    {
        // Remove maximum execution time
        set_time_limit(0);
        $memoryLimit = ini_get('memory_limit');
        ini_set('memory_limit', '5120M');
        $this->scheduleBatches();
        ini_set('memory_limit', $memoryLimit);
    }

    private function scheduleBatches()
    {
        $user = $this->user;
        $batches = [];
        $exportFile = null;

        foreach ($this->getIds() as $key => $ids) {
            if ($key == 0) {
                $exportFile = $this->buildExportFile([array_shift($ids)]);
            }
            $batches[] = (new ExportDataTable($user, $ids))
                ->model($this->model)
                ->resource($this->resource)
                ->included($this->includedFields)
                ->resourceFileName($this->fileName)
                ->format($this->format)
                ->mapping($this->mapping)
                ->exportFile($exportFile);
        }

        $resourceName = $this->fileName;

        Bus::batch($batches)
            ->then(function (Batch $batch) use ($user, $resourceName, $exportFile) {
                // We email the file to the user after successful build of the export file.
                if (! $user) {
                    return;
                }

                // Temporarily increase upload and post size
                $uploadSize = ini_get('upload_max_filesize');
                $postSize = ini_get('post_max_size');

                ini_set('upload_max_filesize', '100M');
                ini_set('post_max_size', '1000M');

                Mail::to($user)
                    ->queue(new DataTableExportMail($user, $resourceName, $exportFile));

                ini_set('upload_max_filesize', $uploadSize);
                ini_set('post_max_size', $postSize);
            })->catch(function (Batch $batch, Throwable $e) use ($user, $resourceName, $exportFile) {
                /**
                 * First batch with failure, we abort and remove the
                 * file as it's corrupt.
                 */
                if ($exportFile && file_exists($exportFile)) {
                    unlink($exportFile);
                }

                // Notify user about failed file if it's the first failure.
                $mail = (new DataTableExportMail($user, $resourceName, $exportFile))->failed();
                Mail::to($user)
                    ->queue($mail);

                /// We cancel the batch.
                $batch->cancel();
            })
            ->name("{$this->fileName} DataTable Export")
            ->onQueue('import-export')
            ->allowFailures(false)
            ->dispatch();
    }

    /**
     * Builds the file to be exported using
     * data for the given ids.
     */
    private function buildExportFile(array $ids = []): string
    {
        $builder = ExportDataTable::buildExportQuery(
            $this->model,
            $ids,
            $this->includedFields
        );

        $records = ExportDataTable::parseResults(
            $builder->get(),
            $this->resource,
            $this->model,
            $this->includedFields
        );

        $exporter = new DataTableExporter(
            $records,
            $this->fileName,
            $this->format,
            $this->user->id,
            $this->mapping,
            $this->includedFields
        );

        return $exporter->export();
    }

    private function getIds(): Generator
    {
        /** @var Builder|mixed $builder */
        $builder = $this->model::with([]);
        $usedTraits = class_uses_recursive($this->model);
        $implementInterfaces = class_implements($this->model);

        $request = [
            'filters' => $this->filters,
            'sortObjs' => $this->sorts,
        ];

        if (in_array(HasFilters::class, $usedTraits) && in_array(Filterable::class, $implementInterfaces)) {
            $builder->filter($request);
            $builder->addRelations($request);
            $builder->addCustomColumns();
        }

        if (in_array(HasSort::class, $usedTraits) && in_array(Sortable::class, $implementInterfaces)) {
            $builder->sort($request);
        }

        if (in_array(Archive::class, $usedTraits)) {
            $builder->archived($this->archived);
        }

        if (! empty($this->extraQueryBindings)) {
            $builder->where(function ($builder) {
                foreach ($this->extraQueryBindings as $binding) {
                    $builder->where($binding['field'], $binding['operator'], $binding['value']);
                }
            });
        }

        /**
         * Fetch ids in chunks
         */
        foreach ($builder->cursor()->chunk(self::EXPORT_DATA_FETCH_PER_PAGE) as $records) {
            yield $records->pluck('id')->toArray();
        }
    }

    public function setIncludedFields(array $included)
    {
        $this->includedFields = $included;

        return $this;
    }

    public function setModel($model)
    {
        $this->model = $model;

        return $this;
    }

    public function setResource($resource)
    {
        $this->resource = $resource;

        return $this;
    }

    public function setFilters(?string $filters)
    {
        $this->filters = $filters;

        return $this;
    }

    public function setSorts(?string $sorts)
    {
        $this->sorts = $sorts ? json_decode($sorts, true) : [];

        return $this;
    }

    public function archived()
    {
        $this->archived = 1;

        return $this;
    }

    public function setExtraQueryBindings(array $bindings)
    {
        $this->extraQueryBindings = $bindings;

        return $this;
    }
}
