<?php

namespace App\DataTable;

use App\Models\AccountingTransaction;
use App\Models\Attribute;
use App\Models\AttributeGroup;
use App\Models\BackorderQueue;
use App\Models\CsvTemplate;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\FinancialAlert;
use App\Models\InventoryAdjustment;
use App\Models\InventoryAssembly;
use App\Models\InventoryMovement;
use App\Models\Payment;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductInventoryMovement;
use App\Models\ProductListing;
use App\Models\ProductPricingTier;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\ReportingDailyFinancial;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\ShipBySchedule;
use App\Models\ShippingMethod;
use App\Models\StockTake;
use App\Models\Store;
use App\Models\StoreEmailTemplate;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\SupplierPricingTier;
use App\Models\SupplierProduct;
use App\Models\Tag;
use App\Models\TaxRate;
use App\Models\TrackedJobLog;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use Illuminate\Contracts\Pagination\Paginator as IPaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonFbaReportCustomerReturn;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Entities\AmazonFbaReportRemovalOrder;
use Modules\Amazon\Entities\AmazonFbaReportRemovalShipment;
use Modules\Amazon\Entities\AmazonFbaReportShipment;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonLedgerSummaryAdjustment;
use Modules\Amazon\Entities\AmazonNewFbaInboundPlan;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoice;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Entities\WooCommerceOrder;
use Modules\WooCommerce\Entities\WooCommerceProduct;
use Modules\Xero\Entities\XeroAccount;

class DataTableConfiguration
{
    /**
     * Get Table Specifications by model name.
     */
    public static function getTableSpecifications($model, bool $withDynamicColumns = true): array
    {
        // TODO: Remove this once we have all models converted to DataTableModelInterface
        $modelName = static::getModelName($model);

        // TODO: Remove conditional once we have all models converted to DataTableModelInterface
        if (isFQN($model) && app($model) instanceof DataTableModelInterface) {
            return [
                'table_specifications' => (new BuildTableSpecifications(app($model)))->build(),
            ];
        }

        // TODO: Remove most of this once we have all models converted to DataTableModelInterface

        $tableSpecifications = config('data_table.'.$modelName);

        if ($model == Product::class || $model == \App\Queries\Product::class) {
            if ($withDynamicColumns) {
                $columns = [];
                $index = 1;
                foreach ($tableSpecifications['columns'] as $column) {
                    if ($column['data_name'] == 'price') {
                        foreach (ProductPricingTier::all() as $pricingTier) {
                            $column['default_order'] = $index++;
                            $column['data_name'] = 'price.'.$pricingTier->name.'.value';
                            $column['column_label'] = 'Pricing: '.$pricingTier->name.' Value';
                            $column['data_type'] = ProductPricingTier::class;
                            $column['data_id'] = $pricingTier->id;
                            $column['default_visible'] = $pricingTier->getRawOriginal('is_default');
                            $column['sortable'] = 1;
                            $column['editable'] = 1;
                            $column['filterable'] = 1;
                            $column['dropdown'] = 0;
                            $column['type'] = 'decimal';
                            $columns[] = $column;

                            $column['default_order'] = $index++;
                            $column['data_name'] = 'price.'.$pricingTier->name.'.currency';
                            $column['column_label'] = 'Pricing: '.$pricingTier->name.' Currency';
                            $column['data_type'] = ProductPricingTier::class;
                            $column['data_id'] = $pricingTier->id;
                            $column['default_visible'] = 0;
                            $column['sortable'] = 0;
                            $column['editable'] = 0;
                            $column['filterable'] = 0;
                            $column['dropdown'] = 1;
                            $column['type'] = 'string';
                            $columns[] = $column;

                            $column['default_order'] = $index++;
                            $column['data_name'] = 'price.'.$pricingTier->name.'.info';
                            $column['column_label'] = 'Pricing: '.$pricingTier->name;
                            $column['data_type'] = ProductPricingTier::class;
                            $column['data_id'] = $pricingTier->id;
                            $column['default_visible'] = 0;
                            $column['sortable'] = 0;
                            $column['editable'] = 0;
                            $column['filterable'] = 0;
                            $column['dropdown'] = 0;
                            $column['type'] = 'currency';
                            $columns[] = $column;
                        }
                    } elseif ($column['data_name'] == 'supplier_pricing') {
                        foreach (SupplierPricingTier::all() as $pricingTier) {
                            $column['default_order'] = $index++;
                            $column['data_name'] = 'supplier_pricing.'.$pricingTier->name.'.value';
                            $column['column_label'] = 'Supplier Pricing: '.$pricingTier->name.' Value';
                            $column['data_type'] = SupplierPricingTier::class;
                            $column['data_id'] = $pricingTier->id;
                            $column['default_visible'] = $pricingTier->getRawOriginal('is_default');
                            $column['sortable'] = 1;
                            $column['editable'] = 1;
                            $column['filterable'] = 1;
                            $column['dropdown'] = 0;
                            $column['type'] = 'decimal';
                            $columns[] = $column;

                            $column['default_order'] = $index++;
                            $column['data_name'] = 'supplier_pricing.'.$pricingTier->name.'.currency';
                            $column['column_label'] = 'Supplier Pricing: '.$pricingTier->name.' Currency';
                            $column['data_type'] = SupplierPricingTier::class;
                            $column['data_id'] = $pricingTier->id;
                            $column['default_visible'] = 0;
                            $column['sortable'] = 0;
                            $column['editable'] = 0;
                            $column['filterable'] = 0;
                            $column['dropdown'] = 1;
                            $column['type'] = 'string';
                            $columns[] = $column;

                            $column['default_order'] = $index++;
                            $column['data_name'] = 'supplier_pricing.'.$pricingTier->name.'.info';
                            $column['column_label'] = 'Supplier Pricing: '.$pricingTier->name;
                            $column['data_type'] = SupplierPricingTier::class;
                            $column['data_id'] = $pricingTier->id;
                            $column['default_visible'] = 0;
                            $column['sortable'] = 0;
                            $column['editable'] = 0;
                            $column['filterable'] = 0;
                            $column['dropdown'] = 0;
                            $column['type'] = 'currency';
                            $columns[] = $column;
                        }
                    } elseif ($column['data_name'] == 'inventory') {
                        foreach (Warehouse::with('supplier')->get() as $warehouse) {
                            $column['default_order'] = $index++;
                            $column['data_name'] = 'inventory.'.$warehouse->name;
                            $column['column_label'] = $warehouse->supplier ?
                "Supplier Inventory: $warehouse->name" :
                "Inventory: $warehouse->name";
                            $column['data_type'] = Warehouse::class;
                            $column['data_id'] = $warehouse->id;
                            $column['default_visible'] = 0;
                            $column['sortable'] = 1;
                            $column['editable'] = 0;
                            $column['filterable'] = 1;
                            $column['dropdown'] = 0;
                            $column['type'] = 'integer';
                            $columns[] = $column;
                        }
                    } elseif ($column['data_name'] == 'inventory_available_warehouses') {
                        foreach (Warehouse::where('type', Warehouse::TYPE_DIRECT)->get() as $warehouse) {
                            $column['default_order'] = $index++;
                            $column['data_name'] = 'inventory_available_warehouses.'.$warehouse->name;
                            $column['column_label'] = "Inventory Available: $warehouse->name";
                            $column['data_type'] = Warehouse::class;
                            $column['data_id'] = $warehouse->id;
                            $column['default_visible'] = 0;
                            $column['sortable'] = 1;
                            $column['editable'] = 0;
                            $column['filterable'] = 1;
                            $column['dropdown'] = 0;
                            $column['type'] = 'integer';
                            $columns[] = $column;
                        }
                    } elseif ($column['data_name'] == 'attributes') {
                        foreach (Attribute::all() as $attribute) {
                            $column['default_order'] = $index++;
                            $column['data_name'] = 'attributes.'.$attribute->name;
                            $column['column_label'] = 'Attribute: '.$attribute->name;
                            $column['data_type'] = Attribute::class;
                            $column['data_id'] = $attribute->id;
                            $column['default_visible'] = 0;
                            $column['sortable'] = 1;
                            $column['editable'] = 1;
                            $column['filterable'] = 1;
                            $column['dropdown'] = intval($attribute->has_options);
                            $column['type'] = $attribute->type;
                            $columns[] = $column;
                        }
                    } else {
                        $column['default_order'] = $index++;
                        $columns[] = $column;
                    }
                }

                $tableSpecifications['columns'] = $columns;
            }
        } elseif ($model == 'shopify.product') {
            $columns = $tableSpecifications['columns'];

            /** @var \App\Models\Shopify\ShopifyProduct $withMostImages */
            $withMostImages = \App\Models\Shopify\ShopifyProduct::getWithMostImages();
            if ($withMostImages) {
                $maxImages = $withMostImages->image_count;
                $index = 0;
                for ($i = 0; $i < $maxImages; $i++) {
                    $column['default_order'] = $index++;
                    $column['data_name'] = 'image'.$index;
                    $column['column_label'] = 'Image '.$index;
                    $column['default_visible'] = 0;
                    $column['group'] = 'Basic Info';
                    $column['group_data'] = 'basic_info';
                    $column['sortable'] = 1;
                    $column['editable'] = 0;
                    $column['filterable'] = 1;
                    $column['dropdown'] = 0;
                    $column['type'] = 'image';
                    $columns[] = $column;
                }

                $tableSpecifications['columns'] = $columns;
            }
        } elseif ($model == 'magento.product') {
            $columns = $tableSpecifications['columns'];
            $index = count($columns);
            for ($i = 0; $i < \App\Models\Magento\Product::maxImageCount(); $i++) {
                $column['default_order'] = ++$index;
                $column['data_name'] = 'image'.($i + 1);
                $column['column_label'] = 'Image '.($i + 1);
                $column['default_visible'] = 0;
                $column['group'] = 'Basic Info';
                $column['group_data'] = 'basic_info';
                $column['sortable'] = 0;
                $column['editable'] = 0;
                $column['filterable'] = 1;
                $column['dropdown'] = 0;
                $column['type'] = 'image';
                $columns[] = $column;
            }

            $tableSpecifications['columns'] = $columns;
        }

        return ['table_specifications' => $tableSpecifications];
    }

    /**
     * Map between frontend model and backend data table config name.
     */
    private static function getModelName($model): string
    {
        $modelName = Str::snake(array_slice(explode('\\', $model), -1)[0]);
        $modelName = str_replace(['crm-', 'settings-'], '', $modelName);
        if (! Str::endsWith($modelName, 'data')) {
            $modelName = Str::singular(str_replace('-', '_', $modelName));
        }

        return match ($modelName) {
            'order' => 'sales_order',
            'supplier_dashboard_inventory' => 'supplier_inventory',
            'inventory_warehouse_transfer' => 'warehouse_transfer',
            'product_audit_trail', 'audio_trail_main' => 'inventory_movement',
            'brand' => 'product_brand',
            'category' => 'product_category',
            'pricing_tier' => 'product_pricing_tier',
            'reports_list' => 'generated_report',
            'fulfillment' => 'sales_order_fulfillment',
            'amazon_order' => 'amazon.order',
            'amazon_product' => 'amazon.product',
            'amazon_fba_report_inventory' => 'amazon.fba_report_inventory',
            'amazon_financial_event_group' => 'amazon.financial_event_group',
            'amazon_report_settlement_data' => 'amazon.settlement_data',
            'amazon_fba_report_customer_return' => 'amazon.fba_report_customer_return',
            'amazon_fba_report_shipment' => 'amazon.fba_report_shipment',
            'amazon_fba_report_removal_shipment' => 'amazon.fba_report_removal_shipment',
            'amazon_fba_report_inventory_ledger' => 'amazon.fba_report_inventory_ledger',
            'amazon_fba_inbound_shipment' => 'amazon.fba_inbound_shipment',
            'amazon_integration_instance' => 'amazon.integration_instance',
            'woo_commerce_integration_instance' => 'woo-commerce.integration_instance',
            'accounting_payment' => 'payment',
            'woo_commerce_product' => 'woo-commerce.product',
            'woo_commerce_order' => 'woo-commerce.order',
            'ebay_product' => 'ebay.product',
            'ebay_legacy_product' => 'ebay.legacy_product',
            'ebay_order' => 'ebay.order',
            'sales_order_line_financial' => 'reporting.sales_order_line_financial',
            default => $modelName,
        };
    }

    /**
     * Get real key from "data_name" key.
     *
     * @param  string  $modelClassName the model class for the datatable
     * @param  string|null  $dataName the datatable key that you want to get the real key for it, NULL to return all real keys for the selected model
     * @return array|string by default any value that has the dot symbol, the code will get it from a relation,
     *                      so we return an Array to tell the code the value is not from a relation (if the query has a join between tables),
     *                      or we can return an Array to tell the code the value is from a relation, but it's not from the foreign key column (for BelongsTo relations)
     *                      or when returning all real keys (if the $dataName is null)
     */
    public static function getRealKey(
        string $modelClassName,
        ?string $dataName = null
    ): array|string {
        $mapWithRealKeys = [
            AccountingTransaction::class => [
                'type' => 'type',
                'type.link_type' => 'link_type',
                'type.link_id' => 'link_id',
                'parent_type' => 'parent.type',
                'parent_type.link_type' => 'parent.link_type',
                'parent_type.link_id' => 'parent.link_id',
                'accountingIntegration' => 'accountingIntegration',
                'last_synced_at' => 'accountingIntegration.updated_at',
                'nominal_code' => [
                    'key' => 'accountingTransactionLines.nominalCode.name',
                    'is_relation' => true,
                    'base' => false,
                ],
                'status' => [
                    'key' => 'status.',
                    'is_relation' => true,
                    'base' => true,
                ],
            ],
            SalesOrderLineFinancial::class => [
                'customer' => 'salesOrderLine.salesOrder.customer.name',
                'sales_order' => 'salesOrderLine.salesOrder.sales_order_number',
                'quantity' => 'salesOrderLine.quantity',
                'description' => 'salesOrderLine.description',
                'amount' => 'salesOrderLine.amount',
                'order_date' => 'salesOrderLine.salesOrder.order_date',
                'product' => 'salesOrderLine.product.sku',
                'sku' => 'salesOrderLine.product.sku',
                'profit' => 'profit',
                'total_cost' => 'total_cost',
                'order_status' => 'salesOrderLine.salesOrder.order_status',
                'fulfillment_status' => 'salesOrderLine.salesOrder.fulfillment_status',
                'store_name' => 'salesOrderLine.salesOrder.salesChannel.store.name',
                'brand_name' => 'salesOrderLine.product.brand.name',
                'integration_name' => 'salesOrderLine.salesOrder.salesChannel.integrationInstance.integration.name',
                'integration_instance_name' => 'salesOrderLine.salesOrder.salesChannel.integrationInstance.name',
//                'supplier_name' => 'salesOrderLine.warehouse.supplier.name',
                'supplier_name' => 'salesOrderLine.product.defaultSupplierProduct.supplier.name'
            ],
            Payment::class => [
                'accountingIntegration' => 'accountingIntegration',
                'last_synced_at' => 'accountingIntegration.updated_at',
                'last_error' => 'accountingIntegration.last_error',
                'type' => 'link.accountingTransaction.type',
                'type.name' => 'name',
                'parent_reference' => 'link.accountingTransaction.reference',
                'type.link_type' => 'link_type',
                'type.link_id' => 'link_id',
                'status' => [
                    'key' => 'status.',
                    'is_relation' => true,
                    'base' => true,
                ],
            ],
            ShipBySchedule::class       => [
                'warehouse' => 'warehouse.name',
                'warehouse.name' => 'warehouse.name',
                'warehouse.id' => 'warehouse_id',
                'ship_by_date' => 'salesOrder.ship_by_date',
                'order_status' => 'salesOrder.order_status',
                'product_sku' => 'product.sku',
            ],
            XeroAccount::class => [
                'nominal_code_name' => 'nominalCode.name',
            ],
            ReportingDailyFinancial::class => [
                'product' => ['key' => 'product.name',
                    'is_relation' => true,
                    'base' => false,
                ],
                'reportable' => ['key' => 'reportable',
                    'is_relation' => false,
                    'base' => false,
                    'is_calculated_column' => true,
                    'name' => 'reportable',
                ],
            ],
            Product::class => [
                'image' => 'attributes.image',
                'image_url' => 'attributes.image',
                'brand' => ['key' => 'brand.name', 'is_relation' => true, 'base' => false],
                'brand.name' => ['key' => 'brand.name', 'is_relation' => true, 'base' => false],
                'tags' => ['key' => 'tags.name', 'is_relation' => true, 'base' => false],
                'tags.name' => [
                    'key' => 'tags.name->en',
                    'is_relation' => true,
                    'base' => false,
                    'from_json' => true,
                ],
                'has_fba' => 'productListings.is_fba',
                'parent_product' => 'parent_id',
                'price' => 'defaultPricing.price',
                'default_supplier' => 'defaultSupplierProduct.supplier.name',
                'default_supplier_sku' => 'supplierProducts.supplier_sku',
                'default_supplier_leadtime' => 'defaultSupplierProduct.leadtime',
                'default_supplier_moq' => 'defaultSupplierProduct.minimum_order_quantity',
            ],
            AmazonProduct::class => [
                'product' => [
                    'key' => 'productListing.product',
                    'is_relation' => true,
                    'base' => true,
                ],
                'duplicate_asins' => [
                    'key' => 'DuplicateAsins.',
                    'is_relation' => true,
                    'base' => false,
                ],
                'asin_multiple_products' => [
                    'key' => 'AsinMultipleProducts.',
                    'is_relation' => true,
                    'base' => false,
                ],
                'asin' => 'asin1',
            ],
            EbayLegacyProduct::class => [
                'product' => [
                    'key' => 'productListing.product',
                    'is_relation' => true,
                    'base' => true,
                ],
            ],
            AmazonOrder::class => [
                'sku_sales_order_id' => 'salesOrder.sales_order_number',
                'PurchaseDate' => 'PurchaseDateUtc',
            ],
            WooCommerceOrder::class => [
                'sku_sales_order_id' => 'salesOrder.sales_order_number',
            ],
            ProductListing::class           => [
                'sales_channel' => 'salesChannel.integrationInstance.name',
            ],
            SalesOrder::class => [
                'sales_channel_name' => 'salesChannel.integrationInstance.name',
                'integration_name' => 'salesChannel.integrationInstance.integration.name',
                'integration' => 'salesChannel.integrationInstance.integration.name',
                'store' => 'store.name',
                'payment_date' => 'fully_paid_at',
                'currency_code' => 'currency.code',
                'shipping_method' => 'shippingMethod.full_name',
                'customer_name' => 'customer.name',
                'customer_name.name' => 'customer.name',
                'customer_email' => 'customer.email',
                'customer_company' => 'customer.company',
                'customer_phone' => 'customer.phone',
                'customer_address' => 'customer.address.address1',
                'customer_city' => 'customer.address.city',
                'customer_province' => 'customer.address.province',
                'customer_province_code' => 'customer.address.province_code',
                'customer_zip' => 'customer.address.zip',
                'customer_country' => 'customer.address.country',
                'customer_country_code' => 'customer.address.country_code',
                'customer_reference' => 'sales_order_number',
                'shipping_city' => 'shippingAddress.city',
                'shipping_state_province'=> 'shippingAddress.province',
                'is_product' => 'salesOrderLines.is_product',
                'item_quantity' => 'salesOrderLines.quantity',
                'item_sku' => 'salesOrderLines.product.sku',
                'product_sku' => 'salesOrderLines.product.sku',
                'item_bundle_sku' => 'salesOrderLines.bundle.sku',
                'item_price' => 'salesOrderLines.amount',
                'item_name' => 'salesOrderLines.description',
                'item_brand' => 'salesOrderLines.product.brand.name',
                'item_nominal_code' => 'salesOrderLines.nominalCode.name',
                'item_tax_rate' => ['key' => 'salesOrderLines.tax_rate_id', 'is_relation' => true, 'base' => false],
                'fulfillment_warehouse' => 'salesOrderFulfillments.warehouse_id',
                'fulfillment_warehouse.name' => [
                    'key' => 'salesOrderFulfillments.warehouse.name',
                    'is_relation' => true,
                    'base' => false,
                ],
                'order_fulfillment_status' => 'fulfillment_status',
                'tags' => ['key' => 'tags.name', 'is_relation' => true, 'base' => false],
                'tags.name' => [
                    'key' => 'tags.name->en',
                    'is_relation' => true,
                    'base' => false,
                    'from_json' => true,
                ],
                'has_backorder' => 'salesOrderLines.has_backorder',
                'warehouses' => 'salesOrderLines.warehouse.name',
                'fulfillable' => [
                    'key' => 'fulfillable.salesOrderFulfillments.salesOrderFulfillmentLines',
                    'is_relation' => true,
                    'base' => true,
                ],
                'sourcing' => [
                    'key' => 'sourcing.',
                    'is_relation' => true,
                    'base' => true,
                ],
                'backordered' => [
                    'key' => 'backordered.',
                    'is_relation' => true,
                    'base' => true,
                ],
                'mapping_needed' => [
                    'key' => 'MappingNeeded.',
                    'is_relation' => true,
                    'base' => false,
                ],
                'tax_revenue' => [
                    'key' => 'taxrevenue.',
                    'is_relation' => true,
                    'base' => false,
                ],
                'covered_by_po' => [
                    'key' => 'coveredByPO.',
                    'is_relation' => true,
                    'base' => true,
                ],
            ],
            Attribute::class => [
                'display_options' => 'display',
                'option_values' => 'values.value',
                'attribute_group' => 'attributeGroup.name',
            ],
            ProductCategory::class => [
                'parent_name' => 'parent.name',
                'category_tree' => 'root.name',
                'category_tree.id' => 'root_id',
                'category_tree.name' => 'root.name',
            ],
            \App\Queries\Product::class => [
                'image' => 'primaryImage.url',
                'image_url' => 'primaryImage.url',
                'other_images' => ['key' => 'otherImages.url', 'is_relation' => true, 'base' => false],
                'image_resolvable' => 'primaryImage.resolvable',
                'dimensions_unit' => 'dimension_unit',
                'case_dimensions_unit' => 'case_dimension_unit',
                'date_of_return' => ['key' => 'salesCreditLines.salesCreditReturnLines.salesCreditReturn.received_at', 'is_relation' => true, 'base' => false],
                'date_of_sales_credit' => ['key' => 'salesCreditLines.salesCredit.credit_date', 'is_relation' => true, 'base' => false],
                'brand' => ['key' => 'brand.name', 'is_relation' => true, 'base' => false],
                'brand.name' => ['key' => 'brand.name', 'is_relation' => true, 'base' => false],
                'tags' => ['key' => 'tags.name', 'is_relation' => true, 'base' => false],
                'tags.name' => [
                    'key' => 'tags.name->en',
                    'is_relation' => true,
                    'base' => false,
                    'from_json' => true,
                ],
                'component_skus' => [
                    'key' => 'sku',
                    'is_relation' => true,
                    'base' => false,
                    'is_calculated_column' => true,
                    'name' => 'ComponentsChildren',
                ],
                'component_ids' => [
                    'key' => 'id',
                    'is_relation' => true,
                    'base' => false,
                    'is_calculated_column' => true,
                    'name' => 'ComponentsChildren',
                ],
                'component_of_sku' => [
                    'key' => 'sku',
                    'is_relation' => true,
                    'base' => false,
                    'is_calculated_column' => true,
                    'name' => 'ComponentsParents',
                ],
                'component_of_id' => [
                    'key' => 'id',
                    'is_relation' => true,
                    'base' => false,
                    'is_calculated_column' => true,
                    'name' => 'ComponentsParents',
                ],
                //        'attributes.value'           => ['key' => 'productAttributes.attribute.name', 'is_relation' => true, 'base' => false],
                'has_fba' => 'productListings.is_fba',
                'parent_product' => 'parent_id',
                'parent_product.sku' => ['key' => 'parent.sku', 'is_relation' => true, 'base' => false],
                'parent_product.name' => ['key' => 'parent.name', 'is_relation' => true, 'base' => false],
                'default_supplier' => [
                    'key' => 'supplierProducts.supplier.name',
                    'is_relation' => true,
                    'base' => false,
                ],
                'default_supplier.id' => 'supplierProducts.supplier_id',
                'default_supplier.name' => [
                    'key' => 'supplierProducts.supplier.name',
                    'is_relation' => true,
                    'base' => false,
                ],
                'default_supplier_sku' => [
                    'key' => 'supplierProducts.supplier_sku',
                    'is_relation' => true,
                    'base' => false,
                ],
                'default_supplier_leadtime' => [
                    'key' => 'supplierProducts.leadtime',
                    'is_relation' => true,
                    'base' => false,
                ],
                'default_supplier_moq' => [
                    'key' => 'supplierProducts.minimum_order_quantity',
                    'is_relation' => true,
                    'base' => false,
                ],
                'inventory_total' => ['key' => 'products_inventory.inventory_total', 'is_relation' => false],
                'inventory_reserved' => ['key' => 'products_inventory.inventory_reserved', 'is_relation' => false],
                'inventory_incoming' => [
                    'key' => 'InventoryIncoming.InventoryIncoming',
                    'is_relation' => true,
                    'is_calculated_column' => true,
                    'name' => 'InventoryIncoming',
                ],
                'inventory_stock_value' => ['key' => 'products_inventory.inventory_stock_value', 'is_relation' => false],
                'inventory_in_transit' => ['key' => 'products_inventory.inventory_in_transit', 'is_relation' => false],
                'inventory_available' => ['key' => 'products_inventory.inventory_available', 'is_relation' => false],
                'category_main' => ['key' => 'primaryCategory.name', 'is_relation' => true, 'base' => false],
                'category_main.name' => ['key' => 'primaryCategory.name', 'is_relation' => true, 'base' => false],
                'category_main_path' => 'primaryCategory.path',
                'category_main_path.name' => 'primaryCategory.path',
                'category_others' => ['key' => 'otherCategories.name', 'is_relation' => true, 'base' => false],
                'category_others.name' => ['key' => 'otherCategories.name', 'is_relation' => true, 'base' => false],
                'category_others_path' => 'otherCategories.path',
                'category_others_path.name' => 'otherCategories.path',
                'sales_nominal_code' => 'salesNominalCode.name',
                'cogs_nominal_code' => 'cogsNominalCode.name',
            ],
            Warehouse::class => [
                'email' => 'addresses.email',
                'phone' => 'addresses.phone',
                'company_name' => 'addresses.company',
                'address1' => 'addresses.address1',
                'address2' => 'addresses.address2',
                'address3' => 'addresses.address3',
                'city' => 'addresses.city',
                'province' => 'addresses.province',
                'province_code' => 'addresses.province_code',
                'zip' => 'addresses.zip',
                'country' => 'addresses.country',
                'country_code' => 'addresses.country_code',
            ],
            WarehouseTransfer::class => [
                'status' => 'transfer_status',
                'from_warehouse' => 'fromWarehouse.name',
                'to_warehouse' => 'toWarehouse.name',
                'shipping_method' => 'shippingMethod.name',
                'item_quantity' => 'warehouseTransferLines.quantity',
                'item_sku' => 'warehouseTransferLines.product.sku',
                'item_name' => 'warehouseTransferLines.product.name',
                'item_quantity_received' => 'shipment.shipmentLines.receiptLines.quantity',
            ],
            InventoryAssembly::class => [
                'warehouse' => 'warehouse.name',
                'item_quantity' => 'inventoryAssemblyLines.quantity',
                'item_sku' => 'inventoryAssemblyLines.product.sku',
                'item_name' => 'inventoryAssemblyLines.product.name',
            ],
            SalesOrderLine::class => [

                'customer' => 'salesOrder.customer.name',
                'sales_order' => 'salesOrder.sales_order_number',
                'product' => 'product.name',
                'sku' => 'product.sku',
                'unit_cost_extended' => 'salesOrderLineFinancial.cogs',
                'order_date' => 'salesOrder.order_date',

            ],
            Supplier::class => [
                'company' => 'addresses.company',
                'address1' => 'addresses.address1',
                'address2' => 'addresses.address2',
                'address3' => 'addresses.address3',
                'city' => 'addresses.city',
                'province' => 'addresses.province',
                'province_code' => 'addresses.province_code',
                'zip' => 'addresses.zip',
                'country' => 'addresses.country',
                'country_code' => 'addresses.country_code',
                'default_shipping_method' => 'defaultShippingMethod',
            ],
            Store::class => [
                'company' => 'company_name',
                'phone' => 'address.phone',
                'company_name' => 'address.company',
                'address1' => 'address.address1',
                'address2' => 'address.address2',
                'address3' => 'address.address3',
                'city' => 'address.city',
                'province' => 'address.province',
                'province_code' => 'address.province_code',
                'zip' => 'address.zip',
                'country' => 'address.country',
                'country_code' => 'address.country_code',
                'sales_channels' => 'salesChannels.integration.name',
                'sales_channels.name' => 'salesChannels.integration.name',
            ],
            Tag::class => [
                'name' => 'name->en', // See https://spatie.be/docs/laravel-tags/v4/advanced-usage/adding-translations
            ],
            CsvTemplate::class => [
                'template_type' => 'model',
                'last_import' => 'last_imported_at',
                'last_export' => 'last_exported_at',
            ],
            AttributeGroup::class => [
                'num_attributes' => 'attributes_count',
                'num_categories' => 'categories_to_attribute_groups_count',
                'num_products' => 'products_count',
            ],
            Customer::class => [
                'sales_channel_origin' => [
                    'key' => 'salesChannelOrigin.integrationInstance.name',
                    'is_relation' => true,
                    'base' => false,
                ],
                'sales_channel_origin.name' => [
                    'key' => 'salesChannelOrigin.integrationInstance.name',
                    'is_relation' => true,
                    'base' => false,
                ],
                'sales_channel_origin.id' => 'sales_channel_origin_id',
                'company_name' => 'company',
                'address2' => 'address.address2',
                'address3' => 'address.address3',
                'city' => 'address.city',
                'province' => 'address.province',
                'province_code' => 'address.province_code',
                'country' => 'address.country',
                'country_code' => 'address.country_code',
            ],
            SalesChannel::class => [
                'integration' => 'integrationInstance.integration.name',
                'integration_image' => 'integrationInstance.integration.image_url',
                'store' => 'store_id',
                'store.id' => 'store_id',
                'store.name' => 'store.name',
            ],
            ShippingMethod::class => [
                'method_name' => 'name',
                'carrier_name' => 'shippingCarrier.name',
                'min_business_days' => 'delivery_min',
                'max_business_days' => 'delivery_max',
            ],
            InventoryMovement::class => [
                'warehouse' => 'warehouse.name',
                'warehouse.name' => 'warehouse.name',
                'warehouse.id' => 'warehouse_id',
                'sku' => 'product.sku',
                'product_name' => 'product.name',
                'barcode' => 'product.barcode',
                'type' => 'type',
                'type.link_type' => 'link_type',
                'type.link_id' => 'link_id',
                'layer' => 'layer_id',
                'layer.layer_type' => 'layer.layer_type',
                'layer.layer_id' => 'layer_id',
                'layer_type' => 'layer.layer_type',
            ],
            ProductInventoryMovement::class => [
                'warehouse' => 'warehouse.name',
                'warehouse.name' => 'warehouse.name',
                'warehouse.id' => 'warehouse_id',
                'type' => 'type',
                'type.link_type' => 'link_type',
                'type.link_id' => 'link_id',
                'layer' => 'layer_id',
                'layer.layer_type' => 'layer.layer_type',
                'layer.layer_id' => 'layer_id',
                'layer_type' => 'layer.layer_type',
            ],
            InventoryAdjustment::class => [
                'warehouse' => 'warehouse.name',
                'warehouse.name' => 'warehouse.name',
                'warehouse.id' => 'warehouse_id',
                'sku' => 'product.sku',
                'type' => ['key' => 'link.link_type', 'is_relation' => true, 'base' => false],
                'reference' => 'link_id',
                'product_name' => 'product.name',
                'product_id' => 'product.name',
                'product' => 'product.name',
            ],
            FifoLayer::class => [
                'sku' => 'product.sku',
                'product_name' => 'product.name',
                'product' => 'product.name',
                'warehouse' => 'warehouse.name',
                'origin' => ['key' => 'link.link_type', 'is_relation' => true, 'base' => false],
            ],
            BackorderQueue::class => [
                'sku' => 'salesOrderLine.product.sku',
                'product_name' => 'salesOrderLine.product.name',
                'sales_order_date' => [
                    'key' => 'salesOrderLine.salesOrder.order_date',
                    'is_relation' => true,
                    'base' => false,
                ],
                'sales_order_number' => 'salesOrderLine.salesOrder.sales_order_number',
                'sales_order' => 'salesOrderLine.salesOrder.sales_order_number',
                'supplier' => 'supplier.name',
                'purchase_order_numbers' => 'backorderQueueCoverages.purchaseOrderLine.purchaseOrder.purchase_order_number',
            ],
            SupplierInventory::class => [
                'sku' => 'product.sku',
                'product_name' => 'product.name',
                'supplier_sku' => [
                    'key' => 'supplier.supplierProducts.supplier_sku',
                    'is_relation' => true,
                    'base' => true,
                ],
                'supplier_id' => 'supplier_id',
                'warehouse' => 'warehouse.name',
                'warehouse_name' => 'warehouse.name',
            ],
            PurchaseOrderLine::class => [
                'sku' => 'product.sku',
                'barcode' => 'product.barcode',
                'is_product' => [
                    'key' => 'product.id',
                    'is_relation' => false,
                    'is_calculated_column' => true,
                    'name' => 'IsProduct',
                ],
                'receipt_status' => [
                    'key' => 'purchaseOrderShipmentReceiptLines.id',
                    'is_relation' => false,
                    'is_calculated_column' => true,
                    'name' => 'ReceiptStatus',
                ],
            ],
            PurchaseOrder::class => [
                'po_number' => 'purchase_order_number',
                'payment_term' => 'paymentTerm.name',
                'incoterm' => 'incoterm.description',
                'shipping_method' => 'requestedShippingMethod.name',
                'requested_shipping_method' => 'requestedShippingMethod.name',
                'shipment_date' => 'purchaseOrderShipments.shipment_date',
                'shipment_tracking' => 'purchaseOrderShipments.tracking',
                'shipment_shipping_method' => 'purchaseOrderShipments.fulfilledShippingMethod.name',
                'supplier_id' => 'supplier.id',
                'supplier_name' => 'supplier.name',
                'supplier_name.name' => 'supplier.name',
                'supplier_name.id' => 'supplier_id',
                'supplier_email' => 'supplier.email',
                'supplier_po_email' => 'supplier.purchase_order_email',
                'supplier_company' => 'supplier.company_name',
                'supplier_contact' => 'supplier.primary_contact_name',
                'supplier_phone' => 'supplier.address.phone',
                'supplier_address' => 'supplier.address.address1',
                'supplier_city' => 'supplier.address.city',
                'supplier_province' => 'supplier.address.province',
                'supplier_province_code' => 'supplier.address.province_code',
                'supplier_zip' => 'supplier.address.zip',
                'supplier_country' => 'supplier.address.country',
                'supplier_country_code' => 'supplier.address.country_code',
                'destination_name' => 'destinationAddress.name',
                'destination_name.name' => 'destinationAddress.name',
                'destination_name.id' => 'destinationAddress.id',
                'destination_email' => 'destinationAddress.email',
                'destination_company' => 'destinationAddress.company',
                'destination_phone' => 'destinationAddress.phone',
                'destination_address' => 'destinationAddress.address1',
                'destination_city' => 'destinationAddress.city',
                'destination_province' => 'destinationAddress.province',
                'destination_province_code' => 'destinationAddress.province_code',
                'destination_zip' => 'destinationAddress.zip',
                'destination_country' => 'destinationAddress.country',
                'destination_country_code' => 'destinationAddress.country_code',
                'quantity' => 'purchaseOrderLines.quantity',
                'sku' => 'purchaseOrderLines.product.sku',
                'supplier_sku' => 'purchaseOrderLines.supplierProducts.supplier_sku',
                'price' => 'purchaseOrderLines.amount',
                'name' => 'purchaseOrderLines.description',
                'tax_rate' => ['key' => 'purchaseOrderLines.tax_rate_id', 'is_relation' => true, 'base' => false],
                'total' => 'total_cost',
                'product_total' => [
                    'key' => 'purchaseOrderLines.subtotal',
                    'is_relation' => true,
                    'base' => false,
                ],
                'nominal_code' => 'purchaseOrderLines.nominalCode.name',
                'eta' => 'purchaseOrderLines.estimated_delivery_date',
                'currency_code' => 'currency.code',
                'tags' => ['key' => 'tags.name', 'is_relation' => true, 'base' => false],
                'tags.name' => [
                    'key' => 'tags.name->en',
                    'is_relation' => true,
                    'base' => false,
                    'from_json' => true,
                ],
                'invoices' => 'purchaseInvoices.supplier_invoice_number',
                'shipment_item_quantity' => 'purchaseOrderShipments.lines_quantity',
            ],

            SupplierProduct::class => [
                'product_name' => 'product.name',
                'product_sku' => 'product.sku',
            ],
            SalesCredit::class => [
                'sales_credit_number' => 'sales_credit_number',
                'store' => 'store.name',
                'store.name' => 'store.name',
                'store.id' => 'store_id',
                'to_warehouse' => 'toWarehouse.name',
                'to_warehouse.name' => 'toWarehouse.name',
                'to_warehouse.id' => 'toWarehouse',
                'returns_receipt_date' => 'fully_returned_at',
                'returns_receipt_items_sku' => 'salesCreditLines.product.sku',
                'returns_receipt_items_quantity' => 'salesCreditLines.quantity',
                'returns_receipt_items_reason' => 'salesCreditLines.salesCreditReturnLines.returnReason.name',
                'returns_receipt_items_reason_notes' => 'salesCreditLines.salesCreditReturnLines.notes',
                'returns_receipt_items_action' => 'salesCreditLines.salesCreditReturnLines.action',
                'customer_name' => 'customer.name',
                'customer_name.name' => 'customer.name',
                'customer_email' => 'customer.email',
                'customer_company' => 'customer.address.company',
                'customer_phone' => 'customer.address.phone',
                'customer_address' => 'customer.address1',
                'customer_city' => 'customer.address.city',
                'customer_province' => 'customer.address.province',
                'customer_province_code' => 'customer.address.province_code',
                'customer_zip' => 'customer.zip',
                'customer_country' => 'customer.address.country',
                'customer_country_code' => 'customer.address.country_code',
                'item_quantity' => 'salesCreditLines.quantity',
                'item_sku' => [
                    'key' => 'salesCreditLines.product.sku',
                    'is_relation' => true,
                    'base' => false,
                ],
                'item_sku.sku' => [
                    'key' => 'salesCreditLines.product.sku',
                    'is_relation' => true,
                    'base' => false,
                ],
                'item_sku.id' => [
                    'key' => 'salesCreditLines.product_id',
                    'is_relation' => true,
                    'base' => false,
                ],
                'item_price' => 'salesCreditLines.amount',
                'item_name' => 'salesCreditLines.description',
                'item_nominal_code' => 'salesCreditLines.nominalCode.name',
                'total' => 'total_credit',
                'currency_code' => 'currency.code',
                'tags' => 'tags.name',
                'has_products' => 'salesCreditLines.has_products',
            ],
            StoreEmailTemplate::class              => [
                'store'      => 'store.name',
                'store.name' => 'store.name',
                'store.id' => 'store_id',
            ],
            StockTake::class => [
                'warehouse' => 'warehouse.name',
                'item_name' => 'stockTakeItems.product.name',
                'item_sku' => 'stockTakeItems.product.sku',
                'item_quantity' => 'stockTakeItems.quantity',
                'item_unit_cost' => 'stockTakeItems.unit_cost',
            ],
            SalesOrderFulfillment::class => [
                'fulfillment_number' => 'salesOrder.fulfillment_number',
                'requested_shipping_method' => 'requestedShippingMethod.full_name',
                'fulfillment_shipping_method' => 'fulfilledShippingMethod.full_name',
                'warehouse' => 'warehouse.name',
                'customer_reference' => 'salesOrder.sales_order_number',
                'store' => 'salesOrder.store.name',
                'sales_channel_type_name' => 'salesOrder.salesChannel.integrationInstance.integration.name',
                'sales_order_shipping_method' => 'salesOrder.shippingMethod.full_name',
                'customer_name' => 'salesOrder.customer.name',
                'customer_email' => 'salesOrder.customer.email',
                'customer_company' => 'salesOrder.customer.address.company',
                'customer_phone' => 'salesOrder.customer.address.phone',
                'customer_address' => 'salesOrder.customer.address.address1',
                'customer_city' => 'salesOrder.customer.address.city',
                'customer_province' => 'salesOrder.customer.address.province',
                'customer_province_code' => 'salesOrder.customer.address.province_code',
                'customer_zip' => 'salesOrder.customer.zip',
                'customer_country' => 'salesOrder.customer.address.country',
                'customer_country_code' => 'salesOrder.customer.address.country_code',
                'item_quantity' => 'salesOrderFulfillmentLines.quantity',
                'item_sku' => 'salesOrderFulfillmentLines.salesOrderLine.product.sku',
                'item_price' => 'salesOrderFulfillmentLines.salesOrderLine.amount',
                'item_name' => 'salesOrderFulfillmentLines.salesOrderLine.description',
                'item_nominal_code' => 'salesOrderFulfillmentLines.salesOrderLine.nominalCode.name',
                'backordered' => ['key' => 'backordered.salesOrderFulfillmentLines', 'is_relation' => true],
                'sales_order_tags' => ['key' => 'salesOrder.tags.name->en', 'is_relation' => true, 'base' => false],
            ],
            GeneratedReport::class                 => [
                'items' => 'generatedReportItems',
            ],
            \App\Models\Magento\Order::class => [
                'state' => ['is_relation' => false, 'key' => 'json_object->state', 'from_json' => true],
                'customer_firstname' => ['is_relation' => false, 'key' => 'json_object->customer_firstname', 'from_json' => true],
                'customer_lastname' => ['is_relation' => false, 'key' => 'json_object->customer_lastname', 'from_json' => true],
                'customer_email' => ['is_relation' => false, 'key' => 'json_object->customer_email', 'from_json' => true],
                'grand_total' => ['is_relation' => false, 'key' => 'json_object->grand_total', 'from_json' => true],
                'created_at' => ['is_relation' => false, 'key' => 'json_object->created_at', 'from_json' => true],
                'updated_at' => ['is_relation' => false, 'key' => 'json_object->updated_at', 'from_json' => true],
            ],
            \App\Models\Magento\Product::class => [
                'product' => 'productListing.product.sku',
            ],
            AmazonFbaReportInventoryLedger::class => [
                'adjustment_made_for_missing_details' => [
                    'key' => 'AdjustmentMadeForMissingDetails.',
                    'is_relation' => true,
                    'base' => false,
                ],
                'sku_link' => [
                    'key' => 'sku_link_type',
                    'is_relation' => false,
                    'base' => false,
                ],
                'sku_reference' => [
                    'key' => 'sku_link_id',
                    'is_relation' => false,
                    'base' => false,
                ],
                'detail_reports' => ['key' => 'detail_type', 'is_relation' => true, 'base' => false],
            ],
            AmazonFbaReportCustomerReturn::class => [
                'ledger_id' => 'ledger.id'
            ],
            AmazonFbaReportShipment::class => [
                'ledger_id' => 'ledger.id'
            ],
            AmazonFbaReportRemovalShipment::class => [
                'ledger_id' => 'ledger.id'
            ],
            WooCommerceProduct::class => [
                'product' => [
                    'key' => 'productListing.product',
                    'is_relation' => true,
                    'base' => true,
                ],
            ],
            AmazonFbaInboundShipment::class => [
                'sku_link' => [
                    'key' => 'sku_link_type',
                    'is_relation' => false,
                    'base' => false,
                ],
            ],
            AmazonNewFbaInboundShipment::class => [
                'sku_link' => [
                    'key' => 'sku_link_type',
                    'is_relation' => false,
                    'base' => false,
                ],
            ],
            AmazonFnskuProduct::class => [
                'product' => 'product.sku'
            ],
            AmazonLedgerSummaryAdjustment::class => [
                'integration_instance_id' => 'amazonFbaReportInventoryLedgerSummary.integration_instance_id',
                'adjustment_date' => 'inventoryAdjustment.adjustment_date',
                'quantity' => 'inventoryAdjustment.quantity',
                'notes' => 'inventoryAdjustment.notes',
                'fnsku' => 'amazonFbaReportInventoryLedgerSummary.fnsku',
                'asin' => 'amazonFbaReportInventoryLedgerSummary.asin',
                'msku' => 'amazonFbaReportInventoryLedgerSummary.msku',
            ],
            FinancialAlert::class => [
                'salesOrder' => 'salesOrderLineFinancial.salesOrderLine.salesOrder.sales_order_number',
                'sku' => 'salesOrderLineFinancial.salesOrderLine.product.sku',
                'total_revenue' => 'salesOrderLineFinancial.total_revenue',
                'total_cost' => 'salesOrderLineFinancial.total_cost',
                'profit' => 'salesOrderLineFinancial.profit',
                'profit_margin' => 'salesOrderLineFinancial.profit_margin',
            ]
        ];

        if ($dataName == null) {
            $dataName = ''; // Convert null to empty string for startsWith check
        }

        if ($modelClassName == \App\Queries\Product::class || $modelClassName == \App\Models\Product::class) {
            if (Str::startsWith($dataName, 'price') || Str::startsWith($dataName, 'supplier_pricing')) {
                $realKey = Str::replaceFirst('price.', 'productPricingTiers.', $dataName);
                $realKey = Str::replaceFirst('supplier_pricing.', 'defaultSupplierProduct.supplierPricingTiers.', $realKey);

                $realKey = Str::replaceLast('.currency', '.currency_code', $realKey);
                $realKey = Str::replaceLast('.value', '.price', $realKey);

                return $realKey;
            }

            if (Str::startsWith($dataName, 'attributes')) {
                $realKey = Str::replaceFirst('attributes', 'productAttributes', $dataName);
                $realKey = implode('.', array_filter(explode('.', $realKey), function ($key, $index) {
                    return $index < 2;
                }, ARRAY_FILTER_USE_BOTH));

                return $realKey;
            }
        }

        // Bind in fields for shopify products
        if ($modelClassName == \App\Models\Shopify\ShopifyProduct::class) {
            $withMostImages = \App\Models\Shopify\ShopifyProduct::getWithMostImages();
            $shopifyImages = [];
            if ($withMostImages) {
                for ($i = 1; $i <= $withMostImages->image_count; $i++) {
                    $shopifyImages["image$i"] = "image.$i";
                }
            }
            $mapWithRealKeys[\App\Models\Shopify\ShopifyProduct::class] = array_merge(
                [
                    'product' => 'productListing.product.sku',
                    'tags' => 'product.tags',
                    'product_type' => 'productListing.product.type',
                ],
                $shopifyImages
            );
        }

        if ($modelClassName == \App\Models\Magento\Product::class) {
            $magentoImages = [];
            for ($i = 0; $i < \App\Models\Magento\Product::maxImageCount(); $i++) {
                $magentoImages['image'.($i + 1)] = ['is_relation' => false, 'key' => "json_object->media_gallery_entries[$i]->file", 'from_json' => true];
            }
            $mapWithRealKeys[\App\Models\Magento\Product::class] = array_merge($mapWithRealKeys[\App\Models\Magento\Product::class] ?? [], $magentoImages);
        }

        if ($dataName) {
            return $mapWithRealKeys[$modelClassName][$dataName] ?? $dataName;
        }

        return array_unique(array_values($mapWithRealKeys[$modelClassName] ?? []));
    }

    /**
     * Get required relations by "data_name" key.
     *
     * @param  null  $dataName
     * @return array|mixed|null
     */
    public static function getRequiredRelations(
        string $modelClassName,
        $dataName = null
    ) {
        $mapWithRelations = [
            AccountingTransaction::class => [
                'last_synced_at' => 'accountingIntegration',
                'type' => 'link',
                'parent_type' => 'parent',
            ],
            Payment::class => [
                'last_synced_at' => 'accountingIntegration',
                'last_error' => 'accountingIntegration',
                'type' => 'link',
            ],
            XeroAccount::class => [
                'nominal_code_name' => 'nominalCode',
            ],
            ReportingDailyFinancial::class => [
                'product' => 'product',
            ],
            Product::class => [
                'image' => 'primaryImage',
                'image_url' => 'primaryImage',
                'image_resolvable' => 'primaryImage',
                'brand' => 'brand',
                'tags' => 'tags',
                'notes' => 'notes',
                'has_fba' => 'productListings',
                'parent_product' => 'parent',
                'variations' => 'variations.suppliers',
                'price' => ['productPricing'],
                'pricing' => ['productPricing', 'productPricing.productPricingTier'],
                'blemished' => 'productBlemished',
                'default_supplier' => [
                    'defaultSupplierProduct',
                    'defaultSupplierProduct.supplier',
                ],
                'default_supplier_sku' => 'defaultSupplierProduct',
                'default_supplier_leadtime' => 'defaultSupplierProduct',
                'default_supplier_moq' => 'defaultSupplierProduct',
                'supplier_pricing' => ['defaultSupplierProduct', 'defaultSupplierProduct.supplierPricingTiers'],
                'inventory_total' => 'totalQuantity',
                'inventory_available' => ['totalQuantity', 'reservedQuantity'],
                'inventory_reserved' => 'reservedQuantity',
                'inventory_incoming' => 'purchaseOrderLines',
                'inventory_in_transit' => 'inTransitQuantity',
                'inventory' => ['inWarehousesQuantity', 'inWarehousesQuantity.warehouse'],
                'inventory_stock_value' => 'inventoryStockValue',
                'daily_average_consumption' => ['salesQuantity', 'dailyMovements', 'totalQuantityBeforePeriod'],
                'attributes' => 'productAttributes',
                'category_main' => ['categories', 'categories.parent'],
                'category_main_path' => ['categories', 'categories.parent'],
                'category_others' => ['categories', 'categories.parent'],
                'category_others_path' => ['categories', 'categories.parent'],
            ],
            SalesOrder::class => [
                'accountingTransaction' => 'accountingTransaction',
                'sales_channel_name' => 'salesChannel.integrationInstance.integration',
                'integration' => 'salesChannel.integrationInstance.integration',
                'store' => ['salesChannel.store', 'store'],
                'currency_code' => 'currency',
                'shipping_method' => 'shippingMethod',
                'customer_email' => 'customer.address',
                'customer_company' => 'customer.address',
                'customer_phone' => 'customer.address',
                'customer_address' => 'customer.address',
                'customer_city' => 'customer.address',
                'customer_province' => 'customer.address',
                'customer_province_code' => 'customer.address',
                'customer_zip' => 'customer.address',
                'customer_country' => 'customer.address',
                'customer_country_code' => 'customer.address',
                'customer_billing_address' => 'customer.billingAddress',
                'sales_order_shipping_address' => 'shippingAddress',
                'sales_order_billing_address' => 'billingAddress',
                'item_info' => [
                    'salesOrderLines',
                    'salesOrderLines.product',
                    'salesOrderLines.product.totalInventory',
                    'salesOrderLines.product.suppliersInventory',
                    'salesOrderLines.product.suppliersInventory',
                    'salesOrderLines.product.primaryImage',
                    'salesOrderLines.nominalCode',
                    'salesOrderLines.warehouse',
                    //          'salesOrderLines.warehouse.supplier',
                ],
                'fulfillment_info' => [
                    'salesOrderFulfillments.warehouse',
                    'salesOrderFulfillments.salesOrderFulfillmentLines',
                    'salesOrderFulfillments.requestedShippingMethod',
                    'salesOrderFulfillments.requestedShippingMethod.shippingCarrier',
                    'salesOrderFulfillments.fulfilledShippingMethod',
                    'salesOrderFulfillments.fulfilledShippingMethod.shippingCarrier',
                ],
                'has_backorder' => 'salesOrderLines.backorderQueue',
                'total_revenue' => 'salesOrderLineFinancials',
                'total_revenue_allocated' => 'salesOrderLineFinancials',
                'tax_revenue' => 'salesOrderLines',
                'product_total' => 'salesOrderLines',
                'total' => ['salesOrderLines', 'financialLines.financialLineType', 'orderLines'],
                'tags' => 'tags',
                'resend' => 'resendLinks',
                'parent_resend' => 'resendParentsLinks.parent',
                'warehouses' => 'salesOrderLines.warehouse.supplier',
                'has_fba_inventory_movements' => 'salesOrderLines',
                'mapping_needed' => 'salesOrderLines',
                'is_fulfilled' => 'salesOrderLines.salesOrderFulfillmentLines',
                'unfulfilled_quantity' => 'salesOrderLines.salesOrderFulfillmentLines',
                'backorder_queue_id' => 'salesOrderLines.backorderQueue',
                'backordered_quantity' => 'salesOrderLines.backorderQueue',
                'returnable_quantity' => 'salesOrderLines.salesCreditLines',
                'purchase_orders' => 'purchaseOrders',
                'sales_credits' => 'salesCredits',
            ],
            Attribute::class => [
                'option_values' => 'values',
                'attribute_group' => 'attributeGroup',
            ],
            ProductCategory::class => [
                'parent_name' => 'parent',
                'category_tree' => 'root',
            ],
            \App\Queries\Product::class => [
                'image' => 'primaryImage',
                'image_url' => 'primaryImage',
                'brand' => 'brand',
                'tags' => 'tags',
                'has_fba' => 'productListings',
                'parent_product' => 'parent',
                'variations' => 'variations',
                'price' => ['productPricingTiers'],
                'default_supplier' => [
                    'defaultSupplierProduct',
                    'defaultSupplierProduct.supplier',
                ],
                'default_supplier_sku' => 'defaultSupplierProduct',
                'default_supplier_leadtime' => 'defaultSupplierProduct',
                'default_supplier_moq' => 'defaultSupplierProduct',
                'default_supplier_price' => ['defaultSupplierProduct.supplierProductPricing', 'defaultSupplierProduct.supplier'],
                'supplier_pricing' => ['defaultSupplierProduct', 'defaultSupplierProduct.supplier', 'defaultSupplierProduct.supplierPricingTiers'],
                'inventory' => ['warehousesInventory', 'warehousesInventory.warehouse'],
                'inventory_available_warehouses' => ['warehousesInventory', 'warehousesInventory.warehouse'],
                'attributes' => 'productAttributeValues.attribute',
                'inventory_incoming' => 'purchaseOrderLines',
                'category_main' => ['primaryCategory'],
                'category_main_path' => ['primaryCategory', 'primaryCategory.parent'],
                'category_others' => ['otherCategories'],
                'category_others_path' => ['otherCategories', 'otherCategories.parent'],
                'sales_nominal_code' => 'salesNominalCode',
                'cogs_nominal_code' => 'cogsNominalCode',
                'components' => 'components',
                'component_skus' => 'components',
                'component_ids' => 'components',
                'component_of_sku' => 'parentProducts',
                'component_of_id' => 'parentProducts',
                'blemished' => 'productBlemished.originalProduct',
                'notes' => 'notes',
            ],
            Warehouse::class => [
                'label' => 'address',
                'email' => 'address',
                'phone' => 'address',
                'company_name' => 'address',
                'address1' => 'address',
                'address2' => 'address',
                'address3' => 'address',
                'city' => 'address',
                'province' => 'address',
                'province_code' => 'address',
                'zip' => 'address',
                'country' => 'address',
                'country_code' => 'address',
                'supplier' => 'supplier',
            ],
            Supplier::class => [
                'email' => 'address',
                'phone' => 'address',
                'company_name' => 'address',
                'address1' => 'address',
                'address2' => 'address',
                'address3' => 'address',
                'city' => 'address',
                'province' => 'address',
                'province_code' => 'address',
                'zip' => 'address',
                'country' => 'address',
                'country_code' => 'address',
                'default_shipping_method' => 'defaultShippingMethod',
            ],
            Store::class => [
                'phone' => 'address',
                'company_name' => 'address',
                'address1' => 'address',
                'address2' => 'address',
                'address3' => 'address',
                'city' => 'address',
                'province' => 'address',
                'province_code' => 'address',
                'zip' => 'address',
                'country' => 'address',
                'country_code' => 'address',
                'sales_channels' => 'salesChannels.integrationInstance',
            ],
            Customer::class => [
                'sales_channel_origin' => 'salesChannelOrigin.integrationInstance',
                'name' => ['billingAddress', 'addresses'],
                'address2' => 'address',
                'address3' => 'address',
                'city' => 'address',
                'province' => 'address',
                'province_code' => 'address',
                'country' => 'address',
                'country_code' => 'address',
            ],
            SalesChannel::class => [
                'name' => 'integrationInstance',
                'integration' => 'integrationInstance.integration',
                'integration_image' => 'integrationInstance.integration',
                'store' => 'store',
            ],
            ShippingMethod::class => [
                'carrier_name' => 'shippingCarrier',
            ],
            InventoryMovement::class => [
                'warehouse' => 'warehouse',
                'type' => 'link',
                'product_name' => 'product',
                'sku' => 'product',
            ],
            InventoryAdjustment::class => [
                'warehouse' => 'warehouse',
                'sku' => 'product',
                'product_name' => 'product',
            ],
            FifoLayer::class => [
                'sku' => 'product',
                'product_name' => 'product',
                'warehouse' => 'warehouse',
            ],
            BackorderQueue::class => [
                'sku' => 'salesOrderLine.product',
                'product_name' => 'salesOrderLine.product',
                'sales_order_number' => 'salesOrderLine.salesOrder',
                'sales_order' => 'salesOrderLine.salesOrder',
                'sales_order_date' => 'salesOrderLine.salesOrder',
                'supplier' => 'supplier',
                'purchase_order_numbers' => 'backorderQueueCoverages.purchaseOrderLine.purchaseOrder',
            ],
            WarehouseTransfer::class => [
                'from_warehouse' => 'fromWarehouse',
                'to_warehouse' => 'toWarehouse',
                'items' => 'warehouseTransferLines',
                'item_quantity' => 'warehouseTransferLines',
                'item_sku' => 'warehouseTransferLines.product',
                'item_name' => 'warehouseTransferLines.product',
                'shipping_method' => 'shippingMethod',
            ],
            InventoryAssembly::class => [
                'warehouse' => 'warehouse',
                'items' => 'inventoryAssemblyLines.inventoryAssembly',
                'item_available_stock' => 'inventoryAssemblyLines.product.inventoryMovements',
                'item_quantity' => 'inventoryAssemblyLines',
                'item_sku' => 'inventoryAssemblyLines.product',
                'item_name' => 'inventoryAssemblyLines.product',
            ],
            SupplierInventory::class => [
                'warehouse' => 'warehouse',
                'sku' => 'product',
                'product_name' => 'product',
                'supplier_sku' => [
                    'product.supplierProducts',
                ],
                'supplier_id' => 'supplier',
            ],
            PurchaseOrderLine::class => [
                'sku' => 'product.primaryImage',
                'barcode' => 'product',
                'supplier_sku' => 'product.supplierProducts',
                'item_inventory' => 'product.totalInventory',
                'supplier_stock' => 'product.suppliersInventory',
                'nominal_code' => 'nominalCode',
                'backorder_coverage' => 'coveredBackorderQueues',
                'backorder_releases' => 'purchaseOrderShipmentReceiptLines',
            ],
            PurchaseOrder::class => [
                'payment_term' => 'paymentTerm',
                'incoterm' => 'incoterm',
                'shipping_method' => ['requestedShippingMethod', 'requestedShippingMethod.shippingCarrier'],
                'requested_shipping_method' => ['requestedShippingMethod', 'requestedShippingMethod.shippingCarrier'],
                'shipment_item_quantity' => 'purchaseOrderShipments.purchaseOrderShipmentLines',
                'shipment_date' => 'purchaseOrderShipments',
                'shipment_tracking' => [
                    'purchaseOrderShipments',
                    'purchaseOrderShipments.fulfilledShippingMethod',
                    'purchaseOrderShipments.fulfilledShippingMethod.shippingCarrier',
                ],
                'shipment_received' => 'purchaseOrderShipments.purchaseOrderShipmentLines.purchaseOrderShipmentReceiptLines',
                'shipment_item_received' => 'purchaseOrderShipments.purchaseOrderShipmentLines.purchaseOrderShipmentReceiptLines',
                'supplier_info' => [
                    'supplier.address',
                ],
                'supplier_id' => 'supplier',
                'supplier_name' => 'supplier',
                'supplier_email' => 'supplier',
                'supplier_po_email' => 'supplier',
                'supplier_company' => 'supplier',
                'supplier_contact' => 'supplier',
                'supplier_phone' => 'supplier.address',
                'supplier_address' => 'supplier.address',
                'supplier_city' => 'supplier.address',
                'supplier_province' => 'supplier.address',
                'supplier_province_code' => 'supplier.address',
                'supplier_zip' => 'supplier.address',
                'supplier_country' => 'supplier.address',
                'supplier_country_code' => 'supplier.address',
                'destination_info' => ['destinationAddress', 'destinationWarehouse'],
                'destination_name' => ['destinationAddress', 'salesOrder'],
                'destination_email' => 'destinationAddress',
                'destination_company' => 'destinationAddress',
                'destination_phone' => 'destinationAddress',
                'destination_address' => 'destinationAddress',
                'destination_city' => 'destinationAddress',
                'destination_province' => 'destinationAddress',
                'destination_province_code' => 'destinationAddress',
                'destination_zip' => 'destinationAddress',
                'destination_country' => 'destinationAddress',
                'destination_country_code' => 'destinationAddress',
                'items' => [
                    'purchaseOrderLines.product.primaryImage',
                    'purchaseOrderLines.product.totalInventory',
                    'purchaseOrderLines.product.supplierProducts',
                    'purchaseOrderLines.purchaseInvoiceLines',
                    'purchaseOrderLines.nominalCode',
                ],
                'currency_code' => 'currency',
                'total' => ['purchaseOrderLines', 'orderLines'], // orderLines is needed due to TaxRateTrait
                'tags' => 'tags',
                'backorder_coverage' => 'purchaseOrderLines.coveredBackorderQueues',
                'backorder_releases' => 'purchaseOrderLines.purchaseOrderShipmentReceiptLines',
            ],

            SupplierProduct::class => [
                'product_name' => 'product',
                'product_sku' => 'product',
            ],
            SalesCredit::class => [
                'customer_reference' => 'salesOrder',
                'store' => 'store',
                'to_warehouse' => 'toWarehouse',
                'returns_receipt_quantity' => ['salesCreditLines', 'salesCreditLines.salesCreditReturnLines'],
                'returns_receipt_items' => [
                    'salesCreditLines',
                    'salesCreditLines.salesCreditReturnLines',
                    'salesCreditLines.salesCreditReturnLines.returnReason',
                ],
                'item_info' => 'salesCreditLines',
                'customer_name' => 'customer',
                'customer_email' => 'customer',
                'customer_company' => 'customer.address',
                'customer_phone' => 'customer.address',
                'customer_address' => 'customer.address',
                'customer_city' => 'customer.address',
                'customer_province' => 'customer.address',
                'customer_province_code' => 'customer.address',
                'customer_zip' => 'customer',
                'customer_country' => 'customer.address',
                'customer_country_code' => 'customer.address',
                'additional_credit' => 'salesCreditLines',
                'tax_credit' => 'salesCreditLines',
                'product_total' => 'salesCreditLines',
                'tags' => 'tags',
            ],
            StoreEmailTemplate::class => [
                'store' => 'store',
                'store.name' => 'store',
            ],
            StockTake::class => [
                'warehouse' => 'warehouse',
                'product' => 'stockTakeItems.product',
                'item_quantity' => 'stockTakeItems',
                'item_sku' => 'stockTakeItems.product',
                'item_name' => 'stockTakeItems.product',
                'image_url' => 'stockTakeItems.product.primaryImage',
            ],
            SalesOrderFulfillment::class => [
                'fulfillment_number' => 'salesOrder',
                'requested_shipping_method' => 'requestedShippingMethod',
                'fulfillment_shipping_method' => 'fulfilledShippingMethod',
                'tracking_number' => ['fulfilledShippingMethod.shippingCarrier', 'salesOrder.salesChannel.integrationInstance.integration'],
                'warehouse' => 'warehouse',
                'customer_reference' => 'salesOrder',
                'store' => 'salesOrder.store',
                'sales_channel_type_name' => 'salesOrder.salesChannel.integrationInstance.integration',
                'sales_order_shipping_method' => 'salesOrder.shippingMethod',
                'customer_name' => 'salesOrder.customer',
                'customer_email' => 'salesOrder.customer',
                'customer_company' => 'salesOrder.customer.address',
                'customer_phone' => 'salesOrder.customer.address',
                'customer_address' => 'salesOrder.customer.address',
                'customer_city' => 'salesOrder.customer.address',
                'customer_province' => 'salesOrder.customer.address',
                'customer_province_code' => 'salesOrder.customer.address',
                'customer_zip' => 'salesOrder.customer',
                'customer_country' => 'salesOrder.customer.address',
                'customer_country_code' => 'salesOrder.customer.address',
                'item_quantity' => 'salesOrderFulfillmentLines',
                'item_sku' => 'salesOrderFulfillmentLines.salesOrderLine.product',
                'item_price' => 'salesOrderFulfillmentLines.salesOrderLine',
                'item_name' => 'salesOrderFulfillmentLines.salesOrderLine',
                'item_nominal_code' => 'salesOrderFulfillmentLines.salesOrderLine.nominalCode',
            ],
            TaxRate::class => [
                'tax_rate_lines' => 'taxRateLines',
            ],
            SalesOrderLine::class => [
                'sales_order' => 'salesOrder',
                'product' => 'product',
            ],
            AmazonFbaInboundShipment::class => [
                'items' => 'amazonFbaInboundShipmentItems',
                'sku_link' => 'skuLink',
            ],
            AmazonNewFbaInboundShipment::class => [
                'items' => 'amazonFbaInboundShipmentItems',
                'sku_link' => 'skuLink',
            ],
            AmazonNewFbaInboundPlan::class => [
                'items' => 'amazonNewFbaInboundPlanItems',
            ],
            AmazonIntegrationInstance::class => [
                'store' => 'salesChannel.store',
                'integration_name' => 'integration',
            ],
            AmazonFbaReportInventoryLedger::class => [
                'details' => 'ledgerDetails',
                'sku_link' => 'skuLink',
                'sku_reference' => 'skuLink',
                'amazon_product' => 'amazonProduct',
                'fnsku_product' => 'amazonFnskuProduct',
            ],
            AmazonFbaReportInventoryLedgerSummary::class => [
                'fnsku_product' => 'amazonFnskuProduct',
            ],
            AmazonFbaReportCustomerReturn::class => [
                'sku' => 'amazonProduct',
            ],
            AmazonFbaReportInventory::class => [
                'sku' => 'amazonProduct',
            ],
            WooCommerceIntegrationInstance::class => [
                'store' => 'salesChannel.store',
                'integration_name' => 'integration',
            ],
            TrackedJobLog::class => [
                'related_model_reference' => 'relatedModel',
            ],
            AmazonLedgerSummaryAdjustment::class => [
                'integration_instance_id' => 'amazonFbaReportInventoryLedgerSummary',
            ],
            ShipMyOrdersInvoice::class => [
                'items' => 'shipMyOrdersInvoiceLines',
            ],
            FinancialAlert::class => [
                'sales_order' => 'salesOrderLineFinancial.salesOrderLine.salesOrder',
                'sku' => 'salesOrderLineFinancial.salesOrderLine.product',
                'total_revenue' => 'salesOrderLineFinancial',
                'total_cost' => 'salesOrderLineFinancial',
                'profit' => 'salesOrderLineFinancial',
                'profit_margin' => 'salesOrderLineFinancial',
                'classification' => 'financialAlertClassification',
                'classification_id' => 'financialAlertClassification',
            ],
        ];

        if ($dataName == null) {
            $dataName = ''; // Convert null to empty string for startsWith check
        }

        if ($modelClassName == \App\Queries\Product::class) {
            if (Str::startsWith($dataName, 'price')) {
                return 'productPricingTiers';
            }

            if (Str::startsWith($dataName, 'supplier_pricing')) {
                return ['defaultSupplierProduct', 'defaultSupplierProduct.supplierPricingTiers'];
            }
        }

        if ($dataName) {
            return $mapWithRelations[$modelClassName][$dataName] ?? null;
        }

        return array_unique(Arr::flatten($mapWithRelations[$modelClassName] ?? []));
    }

    public static function getRequiredCountRelations(
        string $modelClassName,
        $dataName = null
    ) {
        $mapWithCountRelations = [
            AttributeGroup::class => [
                'num_attributes' => 'attributes',
                'num_categories' => 'categoriesToAttributeGroups',
            ],
        ];

        if ($dataName) {
            return $mapWithCountRelations[$modelClassName][$dataName] ?? null;
        }

        return array_unique(Arr::flatten($mapWithCountRelations[$modelClassName] ?? []));
    }

    /**
     * Get persistent relations for model.
     *
     *
     * @return array|string[]
     */
    public static function getPersistentRelations(string $modelClassName): array
    {
        $persistentRelations = [
            SalesOrder::class => [
                'integration' => 'salesChannel.integrationInstance.integration',
                'integration_instance_id' => 'salesChannel',
            ],
            AccountingTransaction::class => [
                'lines' => 'accountingTransactionLines.nominalCode',
            ],
            AmazonFnskuProduct::class => [
                'product' => 'product',
                'initialInventory' => 'amazonFbaInitialInventory',
            ],
            AmazonFbaReportInventoryLedger::class => [
                'fnsku_product' => 'amazonFnskuProduct',
            ],
            AmazonFbaReportInventoryLedgerSummary::class => [
                'fnsku_product' => 'amazonFnskuProduct',
            ],
            AmazonFbaReportRemovalShipment::class => [
                'ledger' => 'ledger',
            ],
            AmazonFbaReportShipment::class => [
                'ledger' => 'ledger',
            ],
            AmazonFbaReportCustomerReturn::class => [
                'ledger' => 'ledger',
            ],
            AmazonFbaReportRemovalOrder::class => [
                'amazon_product' => 'amazonProduct',
                'fnsku_product' => 'amazonFnskuProduct',
                'sku_link' => 'skuLink.warehouseTransfer',
            ],
            AmazonLedgerSummaryAdjustment::class => [
                'summary' => 'amazonFbaReportInventoryLedgerSummary',
                'adjustment' => 'inventoryAdjustment.product'
            ],
        ];

        return $persistentRelations[$modelClassName] ?? [];
    }

    /**
     * Included columns that are required for export model.
     *
     *
     * @return array|string[]
     */
    public static function getIncludedColumnsForExport(string $modelClassName): array
    {
        // TODO: I think we should define all included columns that are required for export without depending on included columns that get from the request

        $includedColumns = [
            SalesOrder::class => [
                'unfulfilled_quantity',
            ],
        ];

        return $includedColumns[$modelClassName] ?? [];
    }

    /**
     * DataTable Pagination.
     *
     *
     * @return IPaginator|Builder[]
     */
    public static function paginate(Builder $builder, array $query = [])
    {
        $pageName = 'page';

        $page = Paginator::resolveCurrentPage($pageName);

        // Read query params from provided query or revert to
        // request
        if (! empty($query['limit'])) {
            $perPage = $query['limit'];
        } else {
            $perPage = request()->get('limit', 10) ?: $builder->getModel()->getPerPage();
        }

        if (! empty($query['total'])) {
            $withTotal = $query['total'];
        } else {
            $withTotal = request()->get('total', 0);
        }

        // without paginate, all data
        if ($perPage == -1) {
            return $builder->get();
        }

        // send meta without "total", only "simplePaginate"
        if ($withTotal == 0) {
            return $builder->simplePaginate($perPage, ['*'], $pageName, $page)->appends($query ?? request()->query());
        }

        // send meta only without data
        if ($withTotal == 1) {
            $total = static::getTotal($builder);
            $results = $builder->getModel()->newCollection();
        } else {
            // send meta(with "total") and data
            $results = ($total = $builder->toBase()->getCountForPagination())
        ? $builder->forPage($page, $perPage)->get()
        : $builder->getModel()->newCollection();
        }

        return ( new DataTablePaginator(
            $results,
            $total,
            $perPage,
            $page,
            [
                'path' => Paginator::resolveCurrentPath(),
                'pageName' => $pageName,
            ]
        ) )->appends($query ?? request()->query());
    }

    protected static function getTotal(Builder $builder)
    {
        $query = $builder->cloneWithout(['orders', 'limit', 'offset'])
            ->cloneWithoutBindings(['order']);

        if (! empty($query->toBase()->getRawBindings()['having'])) {
            // The query has having clause, we need to add the count selection.
            $query->selectRaw('count(*) as count');

            return $query->pluck('count')->count();
        } else {
            return $query->toBase()->getCountForPagination();
        }
    }
}
