<?php

namespace App\DataTable;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\DataTable\Exports\DataTableExporter;
use App\DataTable\Exports\ExportBus;
use App\Exceptions\ExportingEmptyDataException;
use App\Exceptions\InvalidDataTableParametersException;
use App\Exporters\Formatters\ExportFormatterFactory;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\SalesOrder;
use App\Queries\Product;
use App\Response;
use Generator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

trait DataTable
{
    /**
     * @return string|Model|HasFilters|HasSort|Archive
     */
    abstract protected function getModel();

    /**
     * @return string|DataTableResource
     */
    abstract protected function getResource();

    public function index(Request $request, array $extraBindings = []): JsonResponse|AnonymousResourceCollection
    {
        try {
            $builder = $this->dataTableIndex($request, $extraBindings);
        } catch (InvalidDataTableParametersException $e) {
            return response()->json([
                'message' => $e->getMessage(),
            ], 400);
        }

        return $builder instanceof Response ? $builder :
            $this->getResource()::collectionWithTableSpecifications($builder, $this->getModel());
    }

    private function dataTableIndex(Request $request, array $extraBindings = [])
    {
        // only return "table_specifications" if its input is "1"
        if ($this->shouldReturnTableSpecificationsOnly($request)) {
            return $this->getResponseWithTableSpecifications();
        }

        // prevent send included and excluded together
        if (! $this->validateIncludedExcluded($request)) {
            return $this->getResponseWithIncludedExcludedError();
        }

        $builder = $this->buildIndexQuery($request);

        if (! empty($extraBindings)) {
            $builder = $builder->where(function ($builder) use ($extraBindings) {
                foreach ($extraBindings as $binding) {
                    $builder->where($binding['field'], $binding['operator'], $binding['value']);
                }
            });
        }

        // paginate with limit per page (default 10)
        $builder = DataTableConfiguration::paginate($builder);

        return $builder;
    }

    public function abstractIndexForIntegration(Request $request, IntegrationInstanceInterface $integrationInstance)
    {
        $builder = $this->dataTableIndexForIntegration($request, $integrationInstance);

        return $builder instanceof Response ? $builder :
            $this->getResource()::collectionWithTableSpecifications($builder, $this->getModel());
    }

    /**
     * @throws ExportingEmptyDataException
     */
    public function abstractExportForIntegration(Request $request, IntegrationInstanceInterface $integrationInstance): BinaryFileResponse|array|string|Response|RedirectResponse
    {
        return $this->export($this->addIntegrationInstanceFilter($request, $integrationInstance));
    }

    public function dataTableIndexForIntegration(Request $request, $integrationInstance)
    {
        return $this->dataTableIndex($this->addIntegrationInstanceFilter($request, $integrationInstance));
    }

    private function addIntegrationInstanceFilter(Request $request, $integrationInstance): Request
    {
        $request->request->set('integration_instance_id', $integrationInstance->id);

        return $request;
    }

    /**
     * @return Archive|HasFilters|HasSort|Model|string|mixed
     */
    public function buildIndexQuery(Request $request)
    {
        // only return "table_specifications" if its input is "1"
        // only return "table_specifications" if its input is "1"
        if ($this->shouldReturnTableSpecificationsOnly($request)) {
            return $this->getResponseWithTableSpecifications();
        }

        // prevent send included and excluded together
        if (! $this->validateIncludedExcluded($request)) {
            return $this->getResponseWithIncludedExcludedError();
        }

        // set default included fields
        $this->setDefaultIncludedColumns($request);

        // enable query log
        $this->enableQueryLog();
        // model query builder.
        $builder = $this->getModel()::with([]);

        $usedTraits = class_uses_recursive($this->getModel());
        $implementInterfaces = class_implements($this->getModel());

        if (in_array(HasFilters::class, $usedTraits)) {
            $builder->addRelations($request);
            $builder->addCustomColumns();
            $builder->filter($request);
        }

        if (in_array(HasSort::class, $usedTraits)) {
            $builder->sort($request);
        }

        if (in_array(Archive::class, $usedTraits)) {
            $builder->archived($request->get('archived', 0));
        }

        // integration_instance_id is a high level filter that we allow as a direct parameter
        if ($integration_instance_id = $request->get('integration_instance_id')) {
            $builder->where('integration_instance_id', $integration_instance_id);
        }

        return $builder;
    }

    private function getRecordsForExport(Request $request, $totalRows, int $perPage = 1000, array $bindings = []): Generator
    {
        $page = 1;
        $totalPages = ceil($totalRows / $perPage);

        while ($page <= $totalPages) {
            $request->merge([
                'total' => 0,
                'limit' => $perPage,
                'page' => $page,
                'to_export' => true,
            ]);
            $records = $this->index($request, $bindings);

            yield json_decode($records->toResponse($request)->content(), true)['data'];
            $page++;
        }
    }

    /**
     * @param  array  $bindings Extra bindings to the query. These are always 'AND'ed to the builder.
     *
     * @throws ExportingEmptyDataException
     */
    public function export(Request $request, array $bindings = []): RedirectResponse|BinaryFileResponse|Response
    {
        $total = $request->query('total');
        // If the request is only for the total
        // number of affected rows, we respond
        // with that.
        try {
            $exportSizeInfo = $this->getExportDataSize($request, $bindings);
        } catch (ExportingEmptyDataException $e) {
            return $this->response->setMessage('You can\'t export empty data');
        }
        $exportSize = $exportSizeInfo['records_count'] * $exportSizeInfo['columns_count'];

        if ($total == 1) {
            return $this->response->addData(['total' => $exportSize]);
        }
        $format = $request->query('format') ?? ExportFormatterFactory::EXPORT_FORMAT_CSV;
        $exportFileName = method_exists($this, 'getExportFilename') ? $this->getExportFileName() : class_basename($this->getModel());
        if (in_array(MapsExportableFields::class, class_implements($this->getModel()))) {
            $exportable = array_map(function ($field) {
                return $field['exported_as'];
            }, $this->getModel()::getExportableFields());
        } else {
            $exportable = [];
        }

        // add included columns that required for export
        $included = json_decode($request->input('included'), true);
        $included = array_unique(array_merge($included, DataTableConfiguration::getIncludedColumnsForExport($this->getModel())));
        $request->merge(['included' => json_encode($included)]);

        if ($exportSize <= DataTableExporter::MAX_SYNC_EXPORT_SIZE) {
            set_time_limit(0);
            // The number of data points is within the range of sync export,
            // we process the export and download right away.
            $parsedRecords = [];

            foreach ($this->getRecordsForExport($request, $exportSizeInfo['records_count'], 1000, $bindings) as $current) {
                $parsedRecords = array_merge($parsedRecords, $current);
            }

            // If the model transforms export data, we let it transform
            if (in_array(TransformsExportData::class, class_implements($this->getModel()))) {
                $parsedRecords = $this->getModel()::transformExportData($parsedRecords);
            }

            $exporter = new DataTableExporter(
                $parsedRecords,
                $exportFileName,
                $format,
                auth()->id(),
                $exportable,
                $included
            );

            return response()->download($exporter->export());
        } else {
            /**
             * The number of data points exceeds max sync export size,
             * we handle the export job in the background. We setup
             * the export in batches using the Export bus.
             */
            $bus = new ExportBus(
                auth()->user(),
                $exportFileName,
                $format,
                $exportable
            );
            $bus->setFilters($request->query('filters'));
            $bus->setExtraQueryBindings($bindings);
            $bus->setModel($this->getModel());
            $bus->setResource($this->getResource());
            $bus->setSorts($request->query('sortObjs'));
            if ($request->get('archived', 0) == 1) {
                $bus->archived();
            }
            $bus->setIncludedFields($included);

            $bus->process();

            return back()->with(['message' => 'You will receive the data attached to an email shortly.']);
        }
    }

    /**
     * Gets the data size for export.
     * @throws ExportingEmptyDataException
     */
    private function getExportDataSize(Request $request, array $bindings = []): array
    {
        // Cache for reset
        $originalLimit = $request->query('limit');
        $originalTotal = $request->query('total');

        // Export size is (number of rows) x (number of columns)

        // Get number of rows
        $request->merge(['total' => 1]);
        $response = $this->index($request, $bindings);
        $affectedRows = $response->resource->total();

        // Get number of columns
        $request->merge(['total' => 0, 'limit' => 1]);
        $records = $this->index($request, $bindings);

        $parsedRecords = json_decode($records->toResponse($request)->content(), true)['data'];
        // If the model transforms export data, we let it transform
        if (in_array(TransformsExportData::class, class_implements($this->getModel()))) {
            $parsedRecords = $this->getModel()::transformExportData($parsedRecords);
        }

        if (in_array(MapsExportableFields::class, class_implements($this->getModel()))) {
            $exportable = array_map(function ($field) {
                return $field['exported_as'];
            }, $this->getModel()::getExportableFields());
        } else {
            $exportable = [];
        }

        $exporter = new DataTableExporter(
            $parsedRecords,
            'temp.csv',
            $request->query('format') ?? ExportFormatterFactory::EXPORT_FORMAT_CSV,
            auth()->id(),
            $exportable,
            json_decode($request->input('included'), true)
        );

        // Process the records to get the final columns to be exported.
        $exporter->makeRecords();
        $exporter->adaptFieldMapping();

        // Reset cache
        $request->merge(['total' => $originalTotal, 'limit' => $originalLimit]);

        return [
            'records_count' => $affectedRows,
            'columns_count' => count(array_keys($exporter->getRecords()[0])),
        ];
    }

    /**
     * Set default included columns if not set.
     *
     * @param  null  $model
     *
     * @throws InvalidDataTableParametersException
     */
    private function setDefaultIncludedColumns(Request $request, $model = null): void
    {
        // set default model to get its table specifications
        if (is_null($model)) {
            $model = $this->getModel();
        }

        $included = $request->input('included', '[]');
        if (is_array($included)) {
            throw new InvalidDataTableParametersException('There can only be one included parameter.  Use a string of arrays as the parameter value instead of multiple parameters.');
        }
        $included = json_decode($included, true);
        $visibleOnly = (bool) $request->input('visible_only', true);

        if (empty($included) && $visibleOnly) {
            $tableColumns = DataTableConfiguration::getTableSpecifications($model)['table_specifications']['columns'];
            $included = collect($tableColumns)->where('default_visible', 1)->pluck('data_name')->all();
            // add default included to request
            $request->merge(['included' => json_encode($included)]);
        }
    }

    protected function shouldReturnTableSpecificationsOnly(Request $request)
    {
        return $request->input('table_specifications') == 1;
    }

    protected function getResponseWithTableSpecifications()
    {
        return $this->response->addData(DataTableConfiguration::getTableSpecifications($this->getModel()));
    }

    protected function validateIncludedExcluded(Request $request)
    {
        // prevent send included and excluded together
        return ! ($request->has('included') && $request->has('excluded'));
    }

    protected function getResponseWithIncludedExcludedError()
    {
        return $this->response->error(Response::HTTP_BAD_REQUEST, [__('messages.failed.not_both_include_exclude')])
            ->setMessage(__('messages.failed.not_both_include_exclude'));
    }

    /**
     * Enable query log for models (debugging).
     */
    private function enableQueryLog()
    {
        $enabledModels = [Product::class, SalesOrder::class];
        // enable query log for debugging
        if (! app()->runningUnitTests() && ! app()->isProduction() && in_array($this->getModel(), $enabledModels)) {
            DB::enableQueryLog();
        }
    }

    public function getIdsFromFilters(
        $modelClass,
        Request $request,
        string $idField = 'id',
        ?IntegrationInstanceInterface $integrationInstance = null
    ): array
    {
        $query = $modelClass::query();

        // Apply filters to query
        $query = $query->filter($request);

        if ($integrationInstance) {
            $query->where('integration_instance_id', $integrationInstance->id);
        }

        // Get the ids from the query
        return $query->pluck($idField)->toArray();
    }

    public function handleIdsAndFilters(Request $request, $modelClass, $idField = 'id', ?IntegrationInstanceInterface $integrationInstance = null): Request
    {
        $model = app($modelClass);
        $request->validate([
            'ids' => 'sometimes|array|min:1',
            'ids.*' => 'integer|exists:' . $model->getTable() . ',' . $idField,
            'filters' => 'sometimes|string',
        ]);

        if ($request->has('filters')) {
            $request->merge([
                'ids' => $this->getIdsFromFilters(
                    modelClass: $modelClass,
                    request: $request,
                    idField: $idField,
                    integrationInstance: $integrationInstance
                )
            ]);
        }

        return $request;
    }
}
