<?php

namespace App\Data;

use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Validation\Validator;
use Spatie\LaravelData\Attributes\Validation\Exists;
use Spatie\LaravelData\Attributes\Validation\Unique;
use Spatie\LaravelData\Data;
use Spatie\LaravelData\Optional;

class CreateShippingProviderIntegrationInstanceData extends Data
{
    public function __construct(
        #[Unique('integration_instances')]
        public string $name,
        #[Exists('integrations', 'id')]
        public int $integration_id,
        public array|Optional $connection_settings,
        public ShippingProviderIntegrationSettingsData|Optional $integration_settings,
        public bool $is_automatic_sync_enabled = false,
    ) {}

    public static function fromRequest(Request $request): self
    {
        return new self(
            name: $request->input('name'),
            integration_id: $request->input('integration_id'),
            connection_settings: $request->input('connection_settings'),
            integration_settings: ShippingProviderIntegrationSettingsData::fromRequest($request),
            is_automatic_sync_enabled: $request->input('is_automatic_sync_enabled') ?? false,
        );
    }

    public static function withValidator(Validator $validator): void
    {
        $validator->after(function (Validator $validator) {
            $attributes = $validator->attributes();
            $warehouseId = $attributes['integration_settings']['linked_warehouse_id'] ?? null;
            if(empty($warehouseId)){
                $validator->addFailure('linked_warehouse', 'VeracoreMustHaveLinkedWarehouse');
            } else {
                // Check if warehouse is direct or 3pl
                $warehouse = Warehouse::query()->findOrFail($warehouseId);
                if(!in_array($warehouse->type, [Warehouse::TYPE_3PL, Warehouse::TYPE_DIRECT])){
                    $validator->addFailure('linked_warehouse', 'ShippingProviderMustHaveLinkedWarehouse');
                }
            }
        });
    }

    public static function messages(): array
    {
        return [
            'shipping_provider_must_have_linked_warehouse' => 'Shipping provider instance must be linked to a direct or 3pl warehouse.',
        ];
    }
}
