<?php

namespace App\Console\Patches\Shopify\Orders\Patches;

use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyOrderMapping;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;

class FixRefundAdjustmentLinesCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:orders:patch:fix-refund-adjustment-lines
                                {--add : add sales credits from Shopify adjustment refunds}
                                {--debug : confirm mapping and deleting}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Add shopify refund adjustments and fix the existing sales credits that have adjustment lines.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        if ($this->option('add')) {
            $this->info('Adding Sales Credits from Shopify Adjustment Refunds...');
            $this->handleAdjustmentRefunds();
        }

        $this->info('fixing the existing Sales Credits that have adjustment lines...');
        $this->fixReturnRefundsWithAdjustments();

        return self::SUCCESS;
    }

    public function handleAdjustmentRefunds()
    {
        $query = ShopifyOrder::with(['salesOrder'])
            ->whereRelation('salesOrder', 'order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->whereHas('orderMappings', function (Builder $builder) {
                $builder->where('link_type', ShopifyOrderMapping::LINE_TYPE_REFUND_ADJUSTMENT)
                    ->whereNull('sku_link_id');
            });

        if (! $count = $query->count()) {
            $this->info('There are no adjustment refunds to add');

            return self::SUCCESS;
        }

        if (! $this->confirm("There are {$count} adjustment refunds to add, sure?")) {
            return self::SUCCESS;
        }

        $debug = $this->option('debug');

        $query->eachById(function (ShopifyOrder $order) use (&$debug) {
            if ($debug) {
                $choice = $this->choice(
                    'Do you want to add a sales credit to Sales Order ID: '.$order->name.'('.$order->sku_sales_order_id.')'.' from an adjustment refund?',
                    ['Yes', 'Skip', 'Cancel', 'Yes All'],
                    0
                );
                // Skip: don't update the sales credit lines
                if ($choice == 'Skip') {
                    return; // continue
                }
                // Cancel: cancel the operation
                if ($choice == 'Cancel') {
                    return false;
                }
                // Yes All: stop debugging mode
                if ($choice == 'Yes All') {
                    $debug = false;
                }
            } else {
                $this->info("adding a sales credit to Sales Order ID: {$order->sku_sales_order_id} from an adjustment refund");
            }

            $this->info(print_r($order->refunds, 1));

            $order->handleAdjustmentRefunds($order->refunds);
        });
    }

    public function fixReturnRefundsWithAdjustments()
    {
        $shopifySalesChannels = SalesChannel::query()->whereHas('integrationInstance', fn ($q) => $q->shopify())->get();

        $query = SalesCredit::with(['shopifyOrderMappings'])
            ->join('sales_credits', function (JoinClause $join) use ($shopifySalesChannels) {
                $join->on('sales_credits.sales_order_id', '=', 'sales_orders.id')
                    ->where('sales_orders.order_status', '!=', SalesOrder::STATUS_DRAFT)
                    ->whereIn('sales_channel_id', $shopifySalesChannels->pluck('id'));
            })
            ->whereHas('shopifyOrderMappings')
            ->whereHas('salesCreditLines', function (Builder $builder) {
                $builder->whereNull('product_id');
            })
            ->where('total_credit', '<', 0)
            ->select('sales_credits.*');

        if (! $count = $query->count()) {
            $this->info('There are no sales credits with negative balance to fix');

            return self::SUCCESS;
        }

        if (! $this->confirm("There are {$count} sales credits with negative balance to fix, sure?")) {
            return self::SUCCESS;
        }

        $debug = $this->option('debug');

        // deduct discrepancies rather than adding them
        $query->eachById(function (SalesCredit $salesCredit) use (&$debug) {
            if ($debug) {
                $choice = $this->choice(
                    "Do you want to update the sales credit {$salesCredit->sales_credit_number} (ID: {$salesCredit->id}), Sales Order ID: ".$salesCredit->salesOrder->sales_order_number.'('.$salesCredit->sales_order_id.')?',
                    ['Yes', 'Skip', 'Cancel', 'Yes All'],
                    0
                );
                // Skip: don't update the sales credit lines
                if ($choice == 'Skip') {
                    return; // continue
                }
                // Cancel: cancel the operation
                if ($choice == 'Cancel') {
                    return false;
                }
                // Yes All: stop debugging mode
                if ($choice == 'Yes All') {
                    $debug = false;
                }
            } else {
                $this->info("fixing sales credit {$salesCredit->sales_credit_number} (ID: {$salesCredit->id}), Sales Order ID: {$salesCredit->sales_order_id}");
            }

            $shopifyOrder = ShopifyOrder::query()->firstWhere('sku_sales_order_id', $salesCredit->sales_order_id);
            $shopifyOrderMapping = $salesCredit->shopifyOrderMappings->where('shopify_order_id', $shopifyOrder->id)->first();
            $shopifyRefund = collect($shopifyOrder->refunds)->firstWhere('id', $shopifyOrderMapping->link_id);

            $salesCredit->salesCreditLines()->whereNull('product_id')->delete();

            $salesCredit->setSalesCreditLines(
                collect($shopifyRefund['order_adjustments'])->map(function ($adjustLine) {
                    return [
                        'quantity' => 1,
                        'description' => $adjustLine['reason'],
                        'tax' => $adjustLine['tax_amount'],
                        'amount' => -$adjustLine['amount'],
                    ];
                }),
                false
            );

            $salesCredit->load('salesCreditLines');

            $salesCredit->setTotal();
        }, 1000, 'sales_credits.id', 'id');
    }
}
