<?php

namespace App\Console\Patches;

use App\Models\InventoryAdjustment;
use App\Models\SalesCredit;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use Exception;
use Facades\App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Throwable;

class RemoveDuplicateReturns extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:remove-duplicate-returns
                            {--p|products=* : Specify products }
                            {--force : Force deletion even with adjustments. }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Removes duplicate sales credit returns.';

    protected array $affectedOrders = [];

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        set_time_limit(0);

        /**
         * First, we handle instances where there
         *  is only one return on the sales channel order
         * but multiple returns.
         */
        try {
            $this->handleSingleReturns();
        } catch (Throwable $e) {
            //          dd($e->getMessage(), $e->getFile(), $e->getLine());
            $this->output->error($e->getMessage());
        } finally {
            $this->output->info(
                implode(',', $this->affectedOrders)
            );
        }

        return 0;
    }

    /**
     * @throws Throwable
     */
    protected function handleSingleReturns(): void
    {
        $query = SalesOrder::with(['salesCredits'])
            ->withCount('salesCredits')
            ->whereHas('childLinks', function (Builder $builder) {
                return $builder->whereHas('child.salesCreditLines.salesCreditReturnLines', function (Builder $builder) {
                    return $builder->whereIn('action', [SalesCreditReturnLine::ACTION_ADD_TO_STOCK, SalesCreditReturnLine::ACTION_AS_BLEMISHED]);
                });
            })
            ->whereHas('shopifyOrder'); // Only available for shopify orders now

        if (! empty($this->option('products'))) {
            $query = $query->whereHas('salesOrderLines', function (Builder $builder) {
                return $builder->whereIn('product_id', $this->option('products'))
                    ->where('no_audit_trail', 0);
            });
        } else {
            $query = $query->whereHas('salesOrderLines', function (Builder $builder) {
                return $builder->where('no_audit_trail', 0);
            });
        }

        $query = $query->having('sales_credits_links_count', '>', 1);

        $query->cursor()->each(function (SalesOrder $salesOrder) {
            $this->handleForSalesOrder($salesOrder);
        });
    }

    /**
     * @throws Throwable
     */
    protected function handleForSalesOrder(SalesOrder $salesOrder): void
    {
        if (! $salesOrder->shopifyOrder) {
            // This is only available for Shopify orders.
            return;
        }

        // We refresh the order
        ShopifyDownloadOrderrefresh($salesOrder->shopifyOrder);
        $salesOrder->refresh();

        $credits = $salesOrder->salesCredits()->orderBy('created_at', 'desc')->get();
        // Get the refunds

        $refunds = collect($salesOrder->shopifyOrder->refunds ?? []);

        if ($refunds->count() == $credits->count()) {
            return;
        }

        $this->output->info("Fixing sales order: $salesOrder->sales_order_number");

        $this->affectedOrders[] = $salesOrder->sales_order_number;

        if (($refundsCount = $refunds->count()) < $credits->count()) {
            $credits->shift($refundsCount);
            $this->deleteCredits($credits);
        } else {
            throw new Exception("Order: $salesOrder->sales_order_number has more refunds than credits.");
        }
    }

    /**
     * @throws Throwable
     */
    protected function deleteCredits(Collection $credits): void
    {
        $hasAdjustments = false;
        /** @var SalesCredit $salesCredit */
        $credits->each(function (SalesCredit $salesCredit) use (&$hasAdjustments) {
            if (! ($hasAdjustments = $this->linesHaveNegativeAdjustments($salesCredit)) ||
            $this->option('force')) {
                $salesCredit->delete();
            }
        });
        if ($hasAdjustments) {
            $this->output->info('Adjustments exist, skipping.');
        }
    }

    protected function linesHaveNegativeAdjustments(SalesCredit $salesCredit): bool
    {
        return $salesCredit->salesCreditLines()
            ->whereHas('product', function (Builder $builder) {
                return $builder->whereHas('inventoryMovements', function (Builder $builder) {
                    return $builder->whereHasMorph('link', InventoryAdjustment::class, function (Builder $builder) {
                        return $builder->where('quantity', '<', 0);
                    });
                });
            })
            ->count() > 0;
    }
}
