<?php

namespace App\Console\Patches;

use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\SupplierProduct;
use App\Repositories\SupplierInventoryRepository;
use Illuminate\Console\Command;

class FixSupplierInventoryData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:suppliers:fix-inventory';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fixes the supplier inventory for supplier products';

    /**
     * @var SupplierInventoryRepository
     */
    private $inventory;

    /**
     * FixSupplierInventoryData constructor.
     */
    public function __construct(SupplierInventoryRepository $inventory)
    {
        parent::__construct();
        $this->inventory = $inventory;
    }

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        // Get suppliers, and initialize inventory
        // for each supplier product.
        $suppliers = Supplier::with(['supplierProducts'])->get();

        // Progress bar
        $bar = $this->output->createProgressBar($suppliers->count());
        $bar->start();

        $suppliers->each(function (Supplier $supplier) use ($bar) {
            // Skip for suppliers who already have inventory set
            $totalInventory = SupplierInventory::with([])->where('supplier_id', $supplier->id)->count();
            if ($supplier->warehouses()->count() * $supplier->supplierProducts()->count() >= $totalInventory) {
                $this->info("\nSupplier {$supplier->name} is good on inventory, skipping...");

                return;
            }
            // Create the supplier inventory for the supplier's products
            // if doesn't already exist.
            $productsBar = $this->output->createProgressBar($supplier->supplierProducts()->count());
            $productsBar->start();
            $supplier->supplierProducts->each(function (SupplierProduct $supplierProduct) use ($productsBar) {
                $this->inventory->initializeInventoryForSupplierProduct($supplierProduct);
                $productsBar->advance();
            });
            $bar->advance();
        });

        $bar->finish();
        $this->info("\nSupplier inventory fully initialized for suppliers.");
    }
}
