<?php

namespace App\Console\Patches;

use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\FifoLayer;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Services\InventoryManagement\InventoryManager;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class FixStockTakesWithDuplicateMovements extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:stock-takes:fix-duplicate-movements
                                {--i|ids=* : Specify stock take ids }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fixes stock takes with duplicate movements.';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $ids = array_unique($this->option('ids'));
        if (empty($ids)) {
            return;
        }

        foreach ($ids as $id) {
            $this->fixStockTakeById($id);
        }

        return 0;
    }

    private function fixStockTakeById(int $id)
    {
        $this->output->info("Fixing stock-take id: $id");

        DB::transaction(function () use ($id) {
            /** @var StockTake $stockTake */
            $stockTake = StockTake::with(['stockTakeItems', 'stockTakeItems.fifoLayers', 'stockTakeItems.fifoLayers.inventoryMovements'])->find($id);
            if (! $stockTake) {
                $this->output->info("Stock take id: $id not found.");

                return;
            }

            if (! $stockTake->isClosed()) {
                $this->output->info("Stock take id: $id not closed/completed.");

                return;
            }

            $stockTake->stockTakeItems()->each(function (StockTakeItem $stockTakeItem) use ($stockTake) {
                $manager = InventoryManager::with(
                    $stockTake->warehouse_id,
                    $stockTakeItem->product
                );

                // Get the fifo layer.
                $fifoLayers = $stockTakeItem->fifoLayers()->orderBy('fulfilled_quantity', 'desc')->get();

                if ($fifoLayers->isEmpty()) {
                    $manager->addToStock(
                        $stockTakeItem->qty_counted - $stockTakeItem->snapshot_inventory,
                        $stockTakeItem
                    );

                    return;
                }

                $totalFifoCount = 0;
                $fifoIds = [];
                foreach ($fifoLayers as $fifoLayer) {
                    $totalFifoCount += $fifoLayer->original_quantity;
                    $fifoIds[] = $fifoLayer->id;

                    if ($totalFifoCount >= $stockTakeItem->qty_counted - $stockTakeItem->snapshot_inventory) {
                        $stockTakeItem
                            ->fifoLayers()
                            ->whereNotIn('id', $fifoIds)
                            ->where('fulfilled_quantity', 0)
                            ->each(function (FifoLayer $layer) {
                                $layer->inventoryMovements()->delete();
                                $layer->delete();
                            });

                        break;
                    }
                }

                dispatch_sync(new UpdateProductsInventoryAndAvgCost([$stockTakeItem->product_id]));
            });
        });
    }
}
