<?php

namespace App\Console\Patches;

use App\Exceptions\OversubscribedFifoLayerException;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\StockTakeItem;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class FixOrphanLayersInMovements extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:layers:fix-orphan-movements
                            {--p|products=* : Products to fix orphan movements for.}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'This command fixes inventory movements with orphan layer (fifo/backorder) ids by re-creating their layers via adjustments.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        set_time_limit(0);

        try {
            $this->fixFifoLayers();

            $this->fixBackorderQueues();
        } catch (\Throwable $e) {
            dd($e->getMessage(), $e->getFile(), $e->getLine());
        }

        return 0;
    }

    private function fixFifoLayers()
    {
        // Get orphaned fifo layers in inventory movements
        // for sales orders, stock takes and inventory adjustments
        $query = InventoryMovement::with(['link', 'layer'])
            ->where('layer_type', FifoLayer::class)
            ->whereNotNull('layer_id')
            ->whereIn('type', [InventoryMovement::TYPE_SALE, InventoryMovement::TYPE_STOCK_TAKE, InventoryMovement::TYPE_ADJUSTMENT])
            ->whereDoesntHaveMorph('layer', FifoLayer::class);

        if (! empty($this->option('products'))) {
            $query = $query->whereIn('product_id', $this->option('products'));
        }

        $affectedMovements = $query->count();
        $this->info("$affectedMovements movements with orphan Fifo Layers.");

        $fifoLayerIds = $query->pluck('layer_id')->unique()->toArray();
        $uniqueLayers = count($fifoLayerIds);
        $this->info("$uniqueLayers unique orphaned layers in movements.");

        foreach ($fifoLayerIds as $key => $fifoLayerId) {
            /** @var InventoryMovement $movement */
            $movement = InventoryMovement::with([])->where('layer_id', $fifoLayerId)
                ->where('layer_type', FifoLayer::class)
                ->firstOrFail();

            DB::beginTransaction();

            try {
                // We clean movements with non-existing links. For instance,
                // if the movement with the orphaned fifo layer is also linked
                // with a deleted sales order line.
                $this->removeMovementsWithOrphanLinks($fifoLayerId);

                // We now create an adjustment for the remaining orphaned quantity
                // for the fifo layer.
                $adjustment = $this->makeAdjustmentForOrphanQuantity(
                    $this->getOrphanQuantity($fifoLayerId),
                    $movement
                );
                // We map orphaned movements with the new fifo layer from the adjustment.
                $this->mapOrphanedMovementsWithAdjustment($fifoLayerId, $adjustment->getFifoLayer());

                $this->info("Fifo Layer: $fifoLayerId resolved to Fifo Layer: {$adjustment->getFifoLayer()->id}");
                DB::commit();
            } catch (\Throwable $e) {
                DB::rollBack();
                dd($e->getMessage(), $e->getFile(), $e->getLine());
            } finally {
                $processed = $key + 1;
                $this->info("{$processed}/$affectedMovements");
            }
        }
    }

    private function removeMovementsWithOrphanLinks(int $orphanFifoLayerId)
    {
        // First, we remove movements on the orphan fifo with non-existing sales
        // order lines.
        InventoryMovement::with([])
            ->where('layer_type', FifoLayer::class)
            ->where('layer_id', $orphanFifoLayerId)
            ->where('link_type', SalesOrderLine::class)
            ->whereDoesntHaveMorph('link', SalesOrderLine::class)
            ->each(function (InventoryMovement $movement) {
                // Delete any fulfillment movement attached to the sales order line.
                /** @var SalesOrderFulfillmentLine $fulfillmentLine */
                $fulfillmentLine = SalesOrderFulfillmentLine::with(['salesOrderFulfillment'])
                    ->where('sales_order_line_id', $movement->link_id)
                    ->first();

                $fulfillmentLine?->deleteWithFulfillmentIfLast(true);

                $movement->delete();
            });

        // Delete affected stock take movements without stock take item
        InventoryMovement::with([])
            ->where('layer_type', FifoLayer::class)
            ->where('layer_id', $orphanFifoLayerId)
            ->where('link_type', StockTakeItem::class)
            ->whereDoesntHaveMorph('link', StockTakeItem::class)
            ->delete();

        // We delete any inventory adjustment with orphan fifo layer.
        // This cancels the effect of creating a positive adjustment.
        InventoryMovement::with([])
            ->where('layer_type', FifoLayer::class)
            ->where('layer_id', $orphanFifoLayerId)
            ->where('link_type', InventoryAdjustment::class)
            ->each(function (InventoryMovement $movement) {
                $movement->link?->delete();
                $movement->delete();
            });
    }

    /**
     * @throws OversubscribedFifoLayerException
     */
    private function makeAdjustmentForOrphanQuantity(int $orphanQuantity, InventoryMovement $affectedMovement): InventoryAdjustment
    {
        $adjustment = new InventoryAdjustment();
        $adjustment->adjustment_date = now();
        $adjustment->product_id = $affectedMovement->product_id;
        $adjustment->warehouse_id = $affectedMovement->warehouse_id;
        $adjustment->quantity = abs($orphanQuantity);
        $adjustment->is_variable_cost = 0;
        $adjustment->save();

        $fifoLayer = new FifoLayer();
        $fifoLayer->fifo_layer_date = $adjustment->adjustment_date;
        $fifoLayer->product_id = $affectedMovement->product_id;
        $fifoLayer->original_quantity = $adjustment->quantity;
        $fifoLayer->fulfilled_quantity = $adjustment->quantity; // This ensures that the fifo doesn't introduce new inventory
        $fifoLayer->total_cost = $adjustment->product->average_cost * $adjustment->quantity;
        $fifoLayer->warehouse_id = $affectedMovement->warehouse_id;
        $fifoLayer->link_id = $adjustment->id;
        $fifoLayer->link_type = InventoryAdjustment::class;
        $fifoLayer->save();

        $inventoryMovement = new InventoryMovement();
        $inventoryMovement->inventory_movement_date = $adjustment->adjustment_date;
        $inventoryMovement->product_id = $adjustment->product_id;
        $inventoryMovement->quantity = $adjustment->quantity;
        $inventoryMovement->type = InventoryMovement::TYPE_ADJUSTMENT;
        $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $inventoryMovement->warehouse_id = $affectedMovement->warehouse_id;
        $inventoryMovement->warehouse_location_id = $affectedMovement->warehouse_location_id;
        $inventoryMovement->fifo_layer = $fifoLayer->id;

        $adjustment->inventoryMovements()->save($inventoryMovement);

        return $adjustment;
    }

    private function getOrphanQuantity(int $orphanFifoLayerId): int
    {
        return abs(InventoryMovement::with([])
            ->where('layer_id', $orphanFifoLayerId)
            ->where('layer_type', FifoLayer::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->sum('quantity'));
    }

    private function mapOrphanedMovementsWithAdjustment(int $orphanFifoLayerId, FifoLayer $fifoLayer)
    {
        InventoryMovement::with([])
            ->where('layer_type', FifoLayer::class)
            ->where('layer_id', $orphanFifoLayerId)
            ->whereIn('link_type', [SalesOrderLine::class, SalesOrderFulfillmentLine::class, StockTakeItem::class])
            ->update([
                'layer_id' => $fifoLayer->id,
                'layer_type' => FifoLayer::class,
            ]);
        SalesOrderLineLayer::with([])
            ->where('layer_id', $orphanFifoLayerId)
            ->where('layer_type', FifoLayer::class)
            ->update(['layer_id' => $fifoLayer->id]);
    }

    private function fixBackorderQueues()
    {
        // Get orphaned backorder queues in inventory movements
        $query = InventoryMovement::with(['link', 'layer'])
            ->where('layer_type', BackorderQueue::class)
            ->whereNotNull('layer_id')
            ->where('type', InventoryMovement::TYPE_SALE)
            ->whereDoesntHaveMorph('layer', BackorderQueue::class);

        if (! empty($this->option('products'))) {
            $query = $query->whereIn('product_id', $this->option('products'));
        }

        $affectedMovements = $query->count();
        $this->info("$affectedMovements movements with orphan Backorder Queues.");

        $backorderQueueIds = $query->pluck('layer_id')->unique()->toArray();
        $uniqueLayers = count($backorderQueueIds);
        $this->info("$uniqueLayers unique orphaned queues in movements.");

        foreach ($backorderQueueIds as $key => $backorderQueueId) {
            /** @var InventoryMovement $movement */
            $movement = InventoryMovement::with([])->where('layer_id', $backorderQueueId)
                ->where('layer_type', BackorderQueue::class)
                ->firstOrFail();

            DB::beginTransaction();

            try {
                // We remove any movements with non-existing sales order line.
                InventoryMovement::with(['link'])
                    ->where('layer_type', BackorderQueue::class)
                    ->where('layer_id', $backorderQueueId)
                    ->whereDoesntHaveMorph('link', SalesOrderLine::class)
                    ->delete();

                // Try to find an existing backorder queue for the sales order line
                // and map it
                $matchingQueue = BackorderQueue::with([])
                    ->where('sales_order_line_id', $movement->link_id)
                    ->first();
                if ($matchingQueue) {
                    $movement->layer_id = $matchingQueue->id;
                    $movement->save();
                }

                DB::commit();
            } catch (\Throwable $e) {
                DB::rollBack();
                dd($e->getMessage(), $e->getFile(), $e->getLine());
            } finally {
                $processed = $key + 1;
                $this->info("{$processed}/$affectedMovements");
            }
        }
    }
}
