<?php

namespace App\Console\Patches;

use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

class FixDuplicateShopifyShippingLines extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:fix-duplicate-shipping-lines';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fixes duplicate shipping lines for shopify sales orders.';

    /**
     * Execute the console command.
     *
     * @throws Exception
     */
    public function handle(): int
    {
        DB::transaction(function () {
            $query = SalesOrder::with(['shopifyOrder'])
                ->whereHas('salesOrderLines', function (Builder $builder) {
                    $builder->selectRaw('sales_order_id, count(sales_order_id) as total_count')
                        ->where('sales_channel_line_id', 'Shipping')
                        ->groupBy('sales_order_id')
                        ->having('total_count', '>', 1);
                });

            $query->cursor()
                ->each(function (SalesOrder $salesOrder) {
                    $this->syncShippingLinesForSalesOrder($salesOrder);
                });
        });

        return 0;
    }

    /**
     * @throws Exception
     */
    private function syncShippingLinesForSalesOrder(SalesOrder $salesOrder)
    {
        if (! $salesOrder->shopifyOrder) {
            // This patch is only for shopify orders.
            return;
        }

        $totalShopifyShippingCost = collect($salesOrder->shopifyOrder->shipping_lines)
            ->sum('price');

        /** @var SalesOrderLine $firstShippingLine */
        $firstShippingLine = $salesOrder->salesOrderLines()
            ->where('sales_channel_line_id', 'Shipping')
            ->firstOrFail();

        $firstShippingLine->amount = $totalShopifyShippingCost;
        $firstShippingLine->save();

        $salesOrder->salesOrderLines()
            ->where('sales_channel_line_id', 'Shipping')
            ->where('id', '!=', $firstShippingLine->id)
            ->delete();
    }
}
