<?php

namespace App\Console\Patches;

use App\Exceptions\InsufficientStockException;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Services\InventoryManagement\InventoryManager;
use Facades\App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;

class FixClosedOrdersWithReservations extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:orders:closed-with-reservations';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fixes closed orders with active reservation movements.';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        set_time_limit(0);

        $query = SalesOrderFulfillmentLine::with(['salesOrderLine', 'salesOrderLine.salesOrder'])
            ->whereHas('salesOrderLine', function (Builder $builder) {
                return $builder->where('no_audit_trail', 0)
                    ->whereNotNull('product_id')
                    ->whereNotNull('warehouse_id')
                    ->whereHas('salesOrder', function (Builder $builder) {
                        return $builder->where('order_status', SalesOrder::STATUS_CLOSED)
                            ->where('fulfillment_status', SalesOrder::FULFILLMENT_STATUS_FULFILLED);
                    });
            })
            ->whereDoesntHave('inventoryMovements');

        $total = $query->count();

        $this->output->info("Total: $total");

        $processed = 0;

        $query->cursor()
            ->each(function (SalesOrderFulfillmentLine $fulfillmentLine) use (&$processed) {
                $this->output->info("Processing order line id: {$fulfillmentLine->salesOrderLine->id} for sales order: {$fulfillmentLine->salesOrderLine->salesOrder->sales_order_number}");

                $positiveReservations = $this->getPositiveReservationMovement($fulfillmentLine->salesOrderLine);
                if ($positiveReservations->count() == 0) {
                    $this->output->info("No positive movement, creating movements for order line id: {$fulfillmentLine->salesOrderLine->id} for sales order: {$fulfillmentLine->salesOrderLine->salesOrder->sales_order_number}");
                    try {
                        $positiveReservations = $this->createMovementsFor($fulfillmentLine->salesOrderLine);
                    } catch (InsufficientStockException $e) {
                        if ($orderLine = $this->refreshOrder($fulfillmentLine->salesOrderLine)) {
                            $positiveReservations = $this->getPositiveReservationMovement($orderLine);
                        } else {
                            return false;
                        }
                    }
                }

                foreach ($positiveReservations as $positiveReservation) {
                    $negativeReservationMovement = $positiveReservation->replicate();
                    $negativeReservationMovement->quantity = -$positiveReservation->quantity;
                    $negativeReservationMovement->link_type = SalesOrderFulfillmentLine::class;
                    $negativeReservationMovement->link_id = $fulfillmentLine->id;
                    $negativeReservationMovement->reference = $fulfillmentLine->salesOrderLine->salesOrder->sales_order_number;
                    $negativeReservationMovement->save();
                }

                dispatch_sync(new UpdateProductsInventoryAndAvgCost([$fulfillmentLine->salesOrderLine->product_id]));

                $processed++;

                return true;
            });

        $this->output->info("Processed $processed of $total");

        return 0;
    }

    protected function refreshOrder(SalesOrderLine $salesOrderLine)
    {
        $salesOrder = $salesOrderLine->salesOrder;

        // This is only available for shopify orders now.
        if (! $salesOrder->shopifyOrder) {
            return false;
        }

        if ($salesOrder->order_status === SalesOrder::STATUS_CLOSED) {
            $salesOrder->order_status = SalesOrder::STATUS_OPEN;
            $salesOrder->save();
        }
        ShopifyDownloadOrder::refresh($salesOrder->shopifyOrder);

        // Create adjustment for any extra quantities that may be needed.
        if ($salesOrderLine->active_backordered_quantity) {
            $this->createPositiveAdjustmentFor($salesOrderLine);
        }

        $salesOrderLine->refresh();

        return $salesOrderLine;
    }

    protected function createPositiveAdjustmentFor(SalesOrderLine $orderLine)
    {
        /** @var InventoryAdjustment $adjustment */
        $adjustment = InventoryAdjustment::with([])
            ->create([
                'product_id' => $orderLine->product_id,
                'quantity' => $orderLine->active_backordered_quantity,
                'adjustment_date' => now(),
                'warehouse_id' => $orderLine->warehouse_id,
                'notes' => 'Fix inventory integrity.',
            ]);
        InventoryManager::with(
            $orderLine->warehouse_id,
            $orderLine->product
        )->addToStock(abs($adjustment->quantity), $adjustment);
    }

    protected function createMovementsFor(SalesOrderLine $orderLine)
    {
        $quantity = max(0, $orderLine->quantity - $orderLine->canceled_quantity);
        if ($quantity == 0 || ! $orderLine->warehouse_id || ! $orderLine->product) {
            return [];
        }

        InventoryManager::with(
            $orderLine->warehouse_id,
            $orderLine->product
        )->takeFromStock($orderLine->quantity, $orderLine, true, $orderLine);

        return $this->getPositiveReservationMovement($orderLine);
    }

    protected function getPositiveReservationMovement(SalesOrderLine $orderLine): Collection
    {
        return $orderLine->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', '>', 0)
            ->where('layer_type', FifoLayer::class)
            ->get();
    }
}
