<?php

namespace App\Console\Commands\Utilities;

use App\Models\ProductImage;
use App\Models\Store;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class PurgeUnusedImages extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:images:purge
                          {--y|year=  : Specify year to check}
                          {--m|month= : Specify month to check}
                          {--u|user=  : Specify user to check}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Purge unused images from filesystem';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        // check options have valid values
        if (! $options = $this->checkOptions()) {
            return;
        }

        // purge month by month
        $this->purge($options, function (string $monthDirectory) {
            // get used images in this month
            $productImages = $this->getUsedImages(ProductImage::getModel(), 'url', $monthDirectory);
            $storeImages = $this->getUsedImages(Store::getModel(), 'logo_url', $monthDirectory);

            $monthUsedImages = array_merge($productImages, $storeImages); // all used images in this month
            $monthImages = Storage::disk('images')->allFiles($monthDirectory); // all images in this month

            // remove unused images from filesystem
            $delete = Storage::disk('images')->delete(array_diff($monthImages, $monthUsedImages));

            $delete ?
        $this->info("{$monthDirectory}: purged successfully") :
        $this->warn("{$monthDirectory}: some files can't deleted");
        });

        $this->info("\nPurging Finished");
    }

    /**
     * Check/Get command options.
     *
     * @return array|bool
     */
    private function checkOptions()
    {
        $user = intval($this->option('user'));
        $year = intval($this->option('year'));
        $month = intval($this->option('month'));

        if ($this->option('user') && $user < 1) {
            $this->error('The user option must be an integer and valid user id');

            return false;
        }

        if ($this->option('year') && $year < 2019) {
            $this->error('The year option must be an integer and greater than 2019');

            return false;
        }

        if ($this->option('month') && ($month > 12 || $month < 1)) {
            $this->error('The month option must be an integer and a valid month');

            return false;
        }

        return [$user, $year, $month];
    }

    /**
     * Purge unused Images month by month.
     */
    private function purge(array $options, callable $purgeFunction)
    {
        [$user, $year, $month] = $options;

        $this->user($user, $year, $month, $purgeFunction);
    }

    /**
     * Check by user.
     */
    private function user($user, $year, $month, $purgeFunction)
    {
        if ($user) {
            $this->year($user, $year, $month, $purgeFunction);
        } else {
            foreach (Storage::disk('images')->directories() as $userDirectory) {
                $this->year($userDirectory, $year, $month, $purgeFunction);
            }
        }
    }

    /**
     * Check by year.
     */
    private function year($user, $year, $month, $purgeFunction)
    {
        if ($year) {
            $this->month($user, $year, $month, $purgeFunction);
        } else {
            foreach (Storage::disk('images')->directories($user) as $yearDirectory) {
                $this->month($user, $yearDirectory, $month, $purgeFunction);
            }
        }
    }

    /**
     * Check and purge by month.
     */
    private function month($user, $year, $month, $purgeFunction)
    {
        $year = is_int($year) ? "$user/$year" : $year; // year path

        if ($month) {
            $month = sprintf('%02d', $month); // format 2 numbers
            $purgeFunction("$year/$month");
        } else {
            foreach (Storage::disk('images')->directories($year) as $monthDirectory) {
                $purgeFunction($monthDirectory);
            }
        }
    }

    private function getUsedImages(Model $model, string $column, string $monthPath): array
    {
        return $model::with([])
            ->where($column, 'like', "{$monthPath}%")
            ->toBase() // to get original values
            ->pluck($column)
            ->map(function ($image) {
                return [$image, $this->addPrefixToImagePath($image)];
            })
            ->flatten()->toArray();
    }

    private function addPrefixToImagePath($imagePath, $prefix = null)
    {
        if (is_null($prefix)) {
            $prefix = config('image.small_image_prefix');
        }

        $dir = dirname($imagePath);
        $imageName = basename($imagePath);

        return $dir.'/'.$prefix.$imageName;
    }
}
