<?php

namespace App\Console\Commands;

use App\Models\Payment;
use App\Models\SalesOrder;
use App\Models\Shopify\ShopifyOrder;
use Illuminate\Console\Command;

class UpdateExternalReferenceForShopifyPayments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'patch:update-external-reference-for-shopify-payments';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update external reference for shopify payments to avoid duplicate payment records';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        ShopifyOrder::query()
            ->join('sales_orders', 'shopify_orders.sku_sales_order_id', '=', 'sales_orders.id')
            ->join('payments', 'sales_orders.id', '=', 'payments.link_id')
            ->whereNotNull('shopify_orders.transactions')
            ->where('payments.link_type', SalesOrder::class)
            ->whereNull('payments.external_reference')
            ->select('shopify_orders.transactions', 'shopify_orders.name', 'sales_orders.id') // Select columns as required
            ->chunk(5000, function ($orders) {
            $orderPayments = Payment::whereIn('link_id', $orders->pluck('id'))
                ->where('link_type', SalesOrder::class)
                ->get();
            $paymentUpdates = collect();
            foreach ($orders as $order) {
                $payments = $orderPayments->where('link_id', $order->id);
                if (!$payments || $payments->count() === 0) {
                    continue;
                }
                $this->info($order->name);
                $payments = $payments->map(function ($payment) {
                    return collect([
                        'id' => $payment->id,
                        'amount' => number_format($payment->amount, 2, '.', ''),
                    ]);
                })->groupBy('amount');
                #dump($payments->toArray());

                $transactions = collect($order->transactions)
                    ->where('kind', 'SALE')
                    ->where('status', 'SUCCESS')->toArray();

                foreach ($transactions as $transaction) {
                    $transactionId = $transaction['id'] ?? null;
                    $extractedId = extractGraphQlId($transactionId);
                    $amount = number_format($transaction['amount'], 2, '.', '');
                    $this->info("Finding matching payments for " . $amount);
                    $matchingPayments = @$payments[$amount];
                    if (!$matchingPayments) {
                        $this->info("No matching payments found for " . $amount);
                        continue;
                    }

                    $paymentUpdates->push([
                        'id' => $matchingPayments->first()['id'],
                        'external_reference' => $extractedId,
                    ]);

                    $payments[$transaction['amount']]->shift();
                }
            }
            #dd($paymentUpdates->toArray());
            batch()->update(new Payment(), $paymentUpdates->toArray(), 'id');
        });
    }
}
