<?php

namespace App\Console\Commands;

use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;

class SyncFulfillmentMovementQuantity extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:fulfillments:sync-quantities
                            {--p|products=* : Products to sync fulfillment quantities for.}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Syncs sales order fulfillment quantity with corresponding inventory movements.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->clearOrphanFulfillmentLines();

        $this->clearFulfillmentsWithoutLines();

        $query = $this->getFulfillmentMovementsQuery()
            ->whereHasMorph('link', SalesOrderFulfillmentLine::class);

        if (! empty($this->option('products'))) {
            $query = $query->whereIn('product_id', $this->option('products'));
        }

        $totalAffected = $query->count();
        $this->output->info("Affected movements: $totalAffected");

        $query->chunk(1000, function ($movements) {
            $results = $movements->groupBy('layer_id');
            foreach ($results as $fifoLayerId => $movements) {
                DB::beginTransaction();
                try {
                    $this->handleForFifoLayer($fifoLayerId, $movements);
                    DB::commit();
                } catch (\Throwable $e) {
                    DB::rollBack();
                    //                    dd($e->getMessage());
                }
            }
        });

        return 0;
    }

    private function clearFulfillmentsWithoutLines()
    {
        SalesOrderFulfillment::with(['salesOrderFulfillmentLines'])
            ->cursor()
            ->each(function (SalesOrderFulfillment $fulfillment) {
                if ($fulfillment->salesOrderFulfillmentLines()->sum('quantity') == 0) {
                    $this->info("Deleting fulfillment {$fulfillment->id} with 0 units.");
                    $fulfillment->delete();
                }
            });
    }

    private function clearOrphanFulfillmentLines()
    {
        $query = $this->getFulfillmentMovementsQuery()
            ->where('link_type', SalesOrderFulfillmentLine::class)
            ->whereDoesntHaveMorph('link', SalesOrderFulfillmentLine::class);

        if (($total = $query->count()) == 0) {
            $this->output->info('No orphan fulfillment movements');

            return;
        }

        $this->output->warning("Purging $total orphan fulfillment movements");
        $query->delete();

        UpdateProductsInventoryAndAvgCost::dispatch($query->pluck('product_id')->values()->toArray());
    }

    private function getFulfillmentMovementsQuery(): Builder
    {
        $query = InventoryMovement::with([])
//            ->whereHas('product.productInventory', function (Builder $builder) {
//                return $builder->where('inventory_reserved', '<', 0)
//                    ->where('warehouse_id', '!=', 0);
//            })
            ->whereHasMorph('layer', FifoLayer::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', '<', 0); // Fulfillment movements

        if (! empty($this->option('products'))) {
            $query = $query->whereIn('product_id', $this->option('products'));
        }

        return $query;
    }

    private function handleForFifoLayer(int $fifoLayerId, Collection $movements)
    {
        $movements->each(function (InventoryMovement $fulfillmentMovement) {
            // The movement quantity must match that on the original reservation
            // on the sales order line.
            /** @var SalesOrderFulfillmentLine $fulfillmentLine */
            $fulfillmentLine = $fulfillmentMovement->link;

            $salesOrder = $fulfillmentLine->salesOrderLine->salesOrder;

            if (! $salesOrder->shopifyOrder) {
                // This is only available for shopify order at the moment.
                $this->output->warning('Skipping non-shopify order, this patch is only available for shopify orders.');

                return;
            }

            $this->syncFulfillmentQtyWithSalesChannel($fulfillmentLine, $salesOrder);
            $this->syncFulfillmentMovementQty($fulfillmentLine);

            $salesOrder->updateFulfillmentStatus();
        });
    }

    protected function syncFulfillmentMovementQty(SalesOrderFulfillmentLine $fulfillmentLine)
    {
        $fulfillmentLineMovements = $fulfillmentLine->inventoryMovements()->get()->groupBy(['layer_id']);

        /**
         * @var int $fifoLayerId
         * @var Collection  $movements */
        foreach ($fulfillmentLineMovements as $fifoLayerId => $movements) {
            /** @var InventoryMovement $movement */
            foreach ($movements as $movement) {
                /** @var InventoryMovement $originalReservation */
                $originalReservation = InventoryMovement::with([])
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                    ->where('link_type', SalesOrderLine::class)
                    ->where('layer_type', FifoLayer::class)
                    ->where('layer_id', $fifoLayerId)
                    ->where('reference', $movement->reference)
                    ->where('type', InventoryMovement::TYPE_SALE)
                    ->where('quantity', '>', 0)
                    ->first();

                if (! $originalReservation) {
                    // Original reservation doesn't exist, we attempt to create it
                    // from the original active movement.
                    $activeMovementQuery = InventoryMovement::with([])
                        ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                        ->where('link_type', SalesOrderLine::class)
                        ->where('layer_type', FifoLayer::class)
                        ->where('layer_id', $fifoLayerId)
                        ->where('type', InventoryMovement::TYPE_SALE)
                        ->where('reference', $movement->reference)
                        ->where('quantity', '<', 0);

                    if ($activeMovementQuery->count() > 1) {
                        // This is ambiguous, manual review needed.
                        $this->error("Movement ID: {$movement->id}, SKU: {$movement->product->sku} Unable to fix non-existing positive reservation movement, please review.");

                        continue;
                    } elseif ($activeMovementQuery->count() == 0) {
                        // There is no active movement, this movement
                        // shouldn't be on the fifo layer. We delete it.
                        $this->info("Removing movement: {$movement->id}");
                        $movement->delete();
                        UpdateProductsInventoryAndAvgCost::dispatch([$fulfillmentLine->salesOrderLine->product_id]);

                        continue;
                    }

                    /** @var InventoryMovement $activeMovement */
                    $activeMovement = $activeMovementQuery->firstOrFail();
                    $originalReservation = $activeMovement->replicate();
                    $originalReservation->quantity = abs($activeMovement->quantity);
                    $originalReservation->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
                    $originalReservation->save();
                }

                if ($originalReservation->quantity < abs($movement->quantity)) {
                    $movement->quantity = -$originalReservation->quantity;
                    $movement->save();

                    UpdateProductsInventoryAndAvgCost::dispatch([$fulfillmentLine->salesOrderLine->product_id]);
                }
            }
        }
    }

    protected function syncFulfillmentQtyWithSalesChannel(SalesOrderFulfillmentLine $fulfillmentLine, SalesOrder $salesOrder)
    {
        // Next, we attempt to sync the fulfillment quantity with that of the sales channel.
        $salesChannelFulfillmentQuery = collect($salesOrder->shopifyOrder->fulfillments)
            ->pluck('line_items')
            ->collapse()
            ->where('sku', $fulfillmentLine->salesOrderLine->product->sku);

        if ($salesChannelFulfillmentQuery->count() != 1) {
            $this->output->warning("Multiple sales channel fulfillments for fulfillment line: $fulfillmentLine->id of product {$fulfillmentLine->salesOrderLine->product_id}, requires manual review.");

            return;
        }

        $salesChannelFulfillmentQty = $salesChannelFulfillmentQuery->sum('quantity');

        // Also, the sales order fulfillment line quantity must match
        // the quantity in the sales channel order's fulfillment.
        if ($salesChannelFulfillmentQty > $fulfillmentLine->quantity) {
            $fulfillmentLine->quantity = $salesChannelFulfillmentQty;
            $fulfillmentLine->save();
            $fulfillmentLine->negateReservationMovements();

            UpdateProductsInventoryAndAvgCost::dispatch([$fulfillmentLine->salesOrderLine->product_id]);
        }
    }
}
