<?php

namespace App\Console\Commands\Starshipit;

use App\Integrations\Starshipit;
use App\Jobs\Starshipit\GetTrackingJob;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Throwable;

class SyncMissingStarShipItFulfillments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:starshipit:sync-missing-starshipit-fulfillments 
                                {--i|ids=* : Specify order ids }
                                {--debug : Run in debug mode}
                                {--s|source= : shopify|starshipit }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync Missing Starshipit Fulfillments';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        //        $ids = explode(',', $this->argument('ids'));
        $order_numbers = [];
        $source = $this->option('source') ? $this->option('source') : 'shopify';

        /*
         * SELECT so.sales_order_number, max(fulfillment_sequence) as currentSequence, count(*) as numFulfillments
         * FROM sales_order_fulfillments AS sof INNER JOIN sales_orders AS so ON so.id = sof.sales_order_id
         * WHERE sof.fulfillment_type = "starshipit" GROUP BY sales_order_id HAVING numFulfillments<currentSequence;
         */
        $ids = null;

        if (! empty($this->option('ids'))) {
            $ids = explode(',', $this->option('ids')[0]);
        }

        if ($source == 'starshipit') {
            $query = SalesOrderFulfillment::with(['salesOrder'])
                ->selectRaw(DB::raw('sales_order_fulfillments.sales_order_id, max(fulfillment_sequence) currentSequence, count(*) numFulfillments')->getValue(DB::getQueryGrammar()))
                ->join('sales_orders AS so', 'so.id', '=', 'sales_order_fulfillments.sales_order_id')
                ->groupBy('sales_order_fulfillments.sales_order_id')
                ->havingRaw('`numFulfillments` < `currentSequence`');

            if ($ids) {
                $query->whereIn('sales_order_fulfillments.sales_order_id', $ids);
            }

            dd('count: '.$query->count());
            /** @var SalesOrderFulfillment $result */
            foreach ($query->get() as $result) {
                if ($this->option('debug') && ! $this->confirm("Fix for order: {$result->salesOrder->sales_order_number} ?")) {
                    continue;
                }
                $this->fixForSalesOrder($result->salesOrder);
                sleep(3);
            }
        } else {
            /*$results = DB::select('
SELECT sho.sku_sales_order_id, so.sales_order_number, sum(sofl.quantity) sku_fulfillments,
sum_array_cells(JSON_EXTRACT(sho.fulfillments, "$[*].line_items[*].quantity")) as sho_fulfillments,
count(sof.id) AS numSkuFulfillments,
JSON_LENGTH(JSON_UNQUOTE(JSON_EXTRACT(sho.fulfillments, "$[*].id"))) AS numShopifyFulfillments,
JSON_EXTRACT(fulfillments, "$[*].status") as statuses
FROM `shopify_orders` sho join sales_orders so on sho.sku_sales_order_id = so.id
INNER JOIN sales_order_fulfillments sof on sho.sku_sales_order_id = sof.sales_order_id AND sof.status = "fulfilled"
INNER JOIN sales_order_fulfillment_lines AS sofl ON sofl.sales_order_fulfillment_id = sof.id
where sho.created_at > "2021-11-02 00:00:00" AND sku_sales_order_id NOT IN (171241,404534,404565, 285560,276453,207284, 217701,358838,239711)' . ($ids ? ' AND sho.sku_sales_order_id IN (' . implode(",", $ids) . ')' : '') . '
group by sho.sku_sales_order_id, so.sales_order_number, sho.fulfillments having sho_fulfillments > sku_fulfillments;
            ');*/

            $results = DB::select('
SELECT sho.sku_sales_order_id, so.sales_order_number, 
sum_array_cells(JSON_EXTRACT(sho.fulfillments, "$[*].line_items[*].quantity")) as sho_fulfillments, 
count(sof.id) AS numSkuFulfillments,
JSON_LENGTH(JSON_UNQUOTE(JSON_EXTRACT(sho.fulfillments, "$[*].id"))) AS numShopifyFulfillments,
JSON_EXTRACT(fulfillments, "$[*].status") as statuses
FROM `shopify_orders` sho join sales_orders so on sho.sku_sales_order_id = so.id
INNER JOIN `sales_order_lines` AS sol ON sol.sales_order_id = so.id
LEFT JOIN sales_order_fulfillments sof on sho.sku_sales_order_id = sof.sales_order_id  AND sof.status = "fulfilled"
where is_product = 1 AND sho.created_at > "2021-11-04 00:00:00" AND sku_sales_order_id NOT IN (171241,404534,404565, 285560,276453, 207284) AND sof.id IS NULL AND so.sales_order_number NOT LIKE "%LGC%"'.($ids ? ' AND sho.sku_sales_order_id IN ('.implode(',', $ids).')' : '').'
group by sho.sku_sales_order_id, so.sales_order_number, sho.fulfillments having sho_fulfillments > 0 AND statuses NOT LIKE "cancelled";
            ');

            foreach ($results as $key => $result) {
                $salesOrder = SalesOrder::find($result->sku_sales_order_id);

                foreach ($salesOrder->salesOrderLines->where('is_product', 0)->where('description', '!=', 'Shipping') as $salesOrderLine) {
                    $this->info('removing '.$salesOrder->sales_order_number);
                    unset($results[$key]);
                }
            }

            foreach ($results as $key => $result) {
                if (str_contains($result->statuses, 'cancelled')) {
                    continue;
                }
                echo $result->sales_order_number.',';
            }

            $this->info(count($results).' orders to be synced');

            $backorders = [];

            foreach ($results as $result) {
                if (str_contains($result->statuses, 'cancelled')) {
                    $this->info('Skipping because of status: '.$result->statuses);

                    continue;
                }

                $this->info('Fixing sales order '.$result->sales_order_number.'('.$result->sku_sales_order_id.')');

                if ($this->option('debug') && ! $this->confirm('Fix for order: '.$result->sales_order_number.'('.$result->sku_sales_order_id.')?')) {
                    continue;
                }
                $salesOrder = SalesOrder::find($result->sku_sales_order_id);

                DB::beginTransaction();
                try {
                    $this->fixForSalesOrder($salesOrder, $result->numShopifyFulfillments, $backorders);
                } catch (\Exception $e) {
                    DB::rollBack();
                    throw new \Exception($e);
                }
                DB::commit();
            }
        }

        $this->info('List of backorders');

        $this->info(implode(',', $backorders));

        return 0;
    }

    protected function fixForSalesOrder(SalesOrder $salesOrder, $numShopifyFulfillments, &$backorders)
    {
        $salesOrder->salesOrderFulfillments()
            ->each(function (SalesOrderFulfillment $fulfillment) {
                // First, we attempt to delete the order in starshipit.
                try {
                    $fulfillment->delete(true);

                    $this->info("Deleted $fulfillment->fulfillment_sequence.");
                } catch (\Exception $e) {
                    // We ignore
                    $this->info($e->getMessage());
                    $this->info('Skipping...');
                }
            });

        foreach ($salesOrder->salesOrderLines->where('is_product', true)->where('quantity', '>', 0) as $salesOrderLine) {
            //print 'unf: ' . $salesOrderLine->unfulfilled_quantity . "\n";
            //print 'abq: ' . $salesOrderLine->active_backordered_quantity . "\n";
            //exit;
            $fulfillableQuantity = max(0, $salesOrderLine->unfulfilled_quantity - $salesOrderLine->active_backordered_quantity);

            if ($fulfillableQuantity == 0) {
                $this->info($salesOrderLine->product->sku.' is backordered, skipping for now');
                $backorders[] = $salesOrderLine->product->sku;
                DB::rollBack();

                return;
            }
        }

        // Next, we import starshipit orders into sku for the sales order.
        $this->importStarshipitOrdersForSalesOrder($salesOrder, $numShopifyFulfillments);
    }

    protected function importStarshipitOrdersForSalesOrder(SalesOrder $salesOrder, $sho_fulfillments)
    {
        for ($sequence = 1; $sequence <= $sho_fulfillments; $sequence++) { // 4 is the max sequence at the time of the patch.
            $this->importForSequence($sequence, $salesOrder);
        }
    }

    protected function importForSequence(int $sequence, SalesOrder $salesOrder): bool
    {
        echo $salesOrder->sales_order_number.'.'.$sequence."\n";

        if ($salesOrder->fulfillment_status == SalesOrder::FULFILLMENT_STATUS_FULFILLED) {
            return false;
        }

        $starshipit = new Starshipit(IntegrationInstance::with([])->starshipit()->firstOrFail());

        $fulfillmentNumber = $salesOrder->sales_order_number.'.'.$sequence;

        // Abort if the fulfillment number already exists in sku.
        $query = $salesOrder->salesOrderFulfillments()->where('fulfillment_sequence', $sequence);
        if ($query->count() > 0) {
            echo "$fulfillmentNumber already exists in SKU, skipping import..."."\n";

            return false;
        }

        $getOrderResponse = $starshipit->getOrder($fulfillmentNumber, 'order_number');

        if ($getOrderResponse->statusCode == 200 && $getOrderResponse->body['success']) {
            /*
             * Now we can use this payload to create the sales order fulfillment linked to starshipit
             */

            $ssiOrder = $getOrderResponse->body['order'];

            /** @var \App\Models\Starshipit\Mysql\Order $starshipitOrder */
            $starshipitOrder = \App\Models\Starshipit\Mysql\Order::with([])->firstOrNew(['order_id' => $ssiOrder['order_id']]);

            $ssiLines = [];

            if (empty($ssiOrder['items'])) {
                dd('SSI Order has no items: '.$ssiOrder['order_id']);

                return false;
            }

            $warehouse_id = null;

            foreach ($ssiOrder['items'] as $item) {
                if (! isset($item['sku'])) {
                    print_r($item);
                    $this->info('Unexpected SSI item payload so skipping this fulfillment...');

                    return false;
                }
                $matchingLine = $salesOrder->getMatchingUnfulfilledSalesOrderLineFromSku($item['sku']);

                if (! $matchingLine) {
                    DB::rollBack();
                    $this->info('No matching line for '.$item['sku'].'.  Skipping for now...');

                    return false;
                }
                $warehouse_id = $matchingLine['warehouse_id'];
                $ssiLines[] = [
                    'quantity' => $item['quantity'],
                    'sales_order_line_id' => $matchingLine['id'],
                ];
            }

            $request = [
                'fulfilled_at' => $ssiOrder['order_date'],
                'fulfillment_lines' => $ssiLines,
                'fulfillment_type' => 'starshipit',
                'metadata' => json_encode(['signature_required' => $ssiOrder['signature_required']]),
                'requested_shipping_method' => $ssiOrder['shipping_method'] ?? '',
                'warehouse_id' => $warehouse_id,
            ];

            $fulfillment = FulfillSalesOrderService::make($salesOrder)->fulfillWithInputs($request, true, false);

            if (! $starshipitOrder->exists) {
                try {
                    $starshipitOrder->fill($ssiOrder);
                    $starshipitOrder->errors = null;
                    $starshipitOrder->sku_fulfillment_id = $fulfillment->id;
                    $starshipitOrder->save();
                } catch (Throwable $exception) {
                    Log::debug("CreateSkuFulfillmentFromStarShipIt, order_id:{$ssiOrder['order_id']}: {$exception->getMessage()}", $exception->getTrace());
                    try {
                        $starshipitOrder->rawOrder = $ssiOrder;
                        $starshipitOrder->errors = $exception->getMessage();
                        $starshipitOrder->save();
                    } catch (Throwable $exception) {
                        Log::debug("CreateSkuFulfillmentFromStarShipIt(saving raw), order_id:{$ssiOrder['order_id']}: {$exception->getMessage()}", $exception->getTrace());
                    }
                }
            } else {
                Log::info('SSI order '.$ssiOrder['order_id'].' already exists, but creating sku fulfillments');
                $starshipitOrder->sku_fulfillment_id = $fulfillment->id;
                $starshipitOrder->save();
            }

            foreach ($ssiOrder['packages'] as $shippedOrder) {
                $trackingInfo = [
                    'status' => 'Manifested', // at least the status is "Manifested" because we are fetched the Shipped orders
                    'carrier_name' => $ssiOrder['carrier_name'],
                    'carrier_service' => $ssiOrder['carrier_service_code'], // "code" because the webhook and tracking API return the "code" only
                    'tracking_number' => $shippedOrder['tracking_number'],
                    'shipment_date' => $shippedOrder['label_created_date'],
                    'tracking_status' => $shippedOrder['delivery_status'] ?? 'Delivered',
                    'received_by' => 'RequestTrackingUpdatesService',
                    'price_breakdown' => $ssiOrder['price_breakdown'] ?? null,
                ];

                // update sales order fulfillment
                GetTrackingJob::addTrackingToSalesOrderFulfillment($fulfillment, $trackingInfo);
            }

            Log::info("Imported fulfillment for $starshipitOrder->sales_order_number (".$ssiOrder['order_id'].')', $request);

            return true;
        }

        return false;
    }
}
