<?php

namespace App\Console\Commands\Shopify;

use App\Console\Commands\AbstractIntegrationInstanceCommand;
use App\Jobs\Shopify\CreateShopifyGetTransactionsBulkOperationJob;
use App\Jobs\Shopify\ShopifyGetOrdersJob;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use App\Repositories\Shopify\ShopifyOrderRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\Bus;
use Throwable;

class SyncMissingOrdersCommand extends AbstractIntegrationInstanceCommand
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:sync-missing-orders {integrationInstanceId?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync missing Shopify orders from past 2 days';

    public function __construct(private readonly ShopifyOrderRepository $orderRepository)
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        $this->integrationInstances()->each(function (IntegrationInstance $integrationInstance) {

            $options = $this->parseCommandOptions($integrationInstance);

            Bus::batch([
                [
                    new ShopifyGetOrdersJob($integrationInstance, $this->parseCommandOptions($integrationInstance)),
                    new CreateShopifyGetTransactionsBulkOperationJob($integrationInstance),
                ],
            ])
                ->name('Shopify Sync Missing Orders for '.$integrationInstance->name)
                ->onQueue('sales-channels')
                ->dispatch();
        });
    }

    protected function parseCommandOptions($integrationInstance): array
    {
        $created_at_range = $this->orderRepository->getShopifyOrderCreatedDateRangeForMissingSequences(
            $integrationInstance,
            Carbon::now()->subDays(5),
            Carbon::now()->subHour()
        );

        return [
            'downloaded_by' => ShopifyOrder::DOWNLOADED_BY_SYNC_MISSING_ORDERS,
            'limit' => 250,
            'created_after' => $created_at_range['created_at_start'],
            'created_before' => $created_at_range['created_at_end'],
        ];
    }
}
