<?php

namespace App\Console\Commands\Shopify;

use App\Console\Commands\GetIntegrationOrders;
use App\Jobs\Shopify\CreateShopifyGetTransactionsBulkOperationJob;
use App\Jobs\Shopify\ShopifyGetOrdersJob;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use Carbon\Carbon;
use Illuminate\Support\Facades\Bus;

class ShopifyGetOrdersCommand extends GetIntegrationOrders
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:get-orders
                                {integrationInstance : The ID of the integration instance}
                                {--createdAfter= : A date used for selecting orders created after (or at) a specified time.  }
                                {--createdBefore= : A date used for selecting orders created before (or at) a specified time.  }
                                {--lastUpdatedAfter= : A date used for selecting orders that were last updated after (or at) a specified time.  }
                                {--lastUpdatedBefore= : A date used for selecting orders that were last updated before (or at) a specified time.  }
                                {--s|status=* : A list of OrderStatus values. Default Unshipped and PartiallyShipped }
                                {--l|limit=250 : A number that indicates the maximum number of orders that can be returned per page. }
                                {--i|ids=* : Specify order ids }
                                {--debug : Do not queue, run inline }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Retrieve Orders from Shopify sales channel';

    /**
     * @throws \Throwable
     */
    public function handle(): void
    {
        if (! $this->checkOptions()) {
            return;
        }

        // get and check integration instance belongs to shopify
        $integrationInstance = $this->argument('integrationInstance') instanceof IntegrationInstance ? $this->argument('integrationInstance') : IntegrationInstance::with(['integration'])->findOrFail($this->argument('integrationInstance'));

        if (! $integrationInstance->isShopify()) {
            $this->error('The integration instance type must be Shopify');

            return;
        }

        $options = $this->getCommandOptions();

        if (isset($options['ids']) || $this->option('debug')) {
            dispatch_sync(new ShopifyGetOrdersJob($integrationInstance, $options));
        } else {
            Bus::batch([
                [
                    new ShopifyGetOrdersJob($integrationInstance, $options),
                    new CreateShopifyGetTransactionsBulkOperationJob($integrationInstance),
                ],
            ])
                ->name('Shopify Orders/Transactions Download for '.$integrationInstance->name)
                ->onQueue('sales-channels')
                ->dispatch();
        }
    }

    protected function getCommandOptions(): array
    {
        $options = ['downloaded_by' => ShopifyOrder::DOWNLOADED_BY_COMMAND];
        if ($this->option('createdAfter')) {
            $options['created_after'] = Carbon::parse($this->option('createdAfter'));
        }
        if ($this->option('createdBefore')) {
            $options['created_before'] = Carbon::parse($this->option('createdBefore'));
        }
        if ($this->option('lastUpdatedAfter')) {
            $options['updated_after'] = Carbon::parse($this->option('lastUpdatedAfter'));
        }

        if ($this->option('lastUpdatedBefore')) {
            $options['updated_before'] = Carbon::parse($this->option('lastUpdatedBefore'));
        }

        if ($this->option('status')) {
            $options['order_statuses'] = $this->option('status');
        }

        if ($this->option('limit')) {
            $options['limit'] = (int) $this->option('limit');
        }

        if ($this->option('ids')) {
            $options['ids'] = implode(',', $this->option('ids'));
        }

        return $options;
    }
}
